package com.particles.android.ads.internal.loader

import android.content.Context
import com.particles.android.ads.*
import com.particles.android.ads.appopen.AppOpenAd
import com.particles.android.ads.internal.AdErrorPool
import com.particles.android.ads.internal.AppOpenAdImpl
import com.particles.android.ads.internal.data.cache.AdCacheImpl
import com.particles.android.ads.internal.domain.Ad
import com.particles.android.ads.internal.domain.AdSession
import com.particles.android.ads.internal.util.Callback
import kotlinx.coroutines.*

class AppOpenAdLoader(
    private val context: Context,
    private val adUnitId: String,
    private val listener: AdLoadListener<AppOpenAd>
) : AdLoader {

    private val cache = AdCacheImpl(context, "nova-ads")

    override fun loadAds(adRequest: AdRequest, maxNumberOfAds: Int) {
        if (/* TODO: ABTestV3.isNovaAppOpenAdOptimization() */ false) {
            loadAdsOptimization(adRequest, maxNumberOfAds)
        } else {
            loadAdsLegacy(adRequest, maxNumberOfAds)
        }
    }

    private fun loadAdsOptimization(adRequest: AdRequest, maxNumberOfAds: Int) {
        if (adRequest.requestOptions.onlyRetrieveFromCache) {
            loadAdsFromCache(maxNumberOfAds, object : Callback<List<Ad>> {
                override fun onSuccess(result: List<Ad>) {
                    val ads = result.map { AppOpenAdImpl(AdSession(adUnitId = adUnitId, adRequest = adRequest, ad = it)) }
                    notifyAdLoaded(ads)
                }

                override fun onFailure(e: Throwable) {
                    val adError = if (e is AdError) e else AdErrorPool.GENERAL_LOAD_ERROR.toAdError(e)
                    notifyAdLoadFailed(adError)
                }
            })
        } else {
            loadAdsFromNetwork(adRequest, maxNumberOfAds, object : Callback<List<Ad>> {
                override fun onSuccess(result: List<Ad>) {
                    saveAdsToCache(result)
                    val ads = result.map { AppOpenAdImpl(AdSession(adUnitId = adUnitId, adRequest = adRequest, ad = it)) }
                    notifyAdLoaded(ads)
                }

                override fun onFailure(e: Throwable) {
                    val adError = if (e is AdError) e else AdErrorPool.GENERAL_LOAD_ERROR.toAdError(e)
                    notifyAdLoadFailed(adError)
                }
            })
        }
    }

    private fun loadAdsLegacy(adRequest: AdRequest, maxNumberOfAds: Int) {
        val loadAdsFromNetworkToCache = {
            loadAdsFromNetwork(adRequest, 3, object : Callback<List<Ad>> {
                override fun onSuccess(result: List<Ad>) {
                    saveAdsToCache(result)
                }

                override fun onFailure(e: Throwable) {

                }
            })
        }

        loadAdsFromCache(maxNumberOfAds, object : Callback<List<Ad>> {
            override fun onSuccess(result: List<Ad>) {
                val ads = result.map { AppOpenAdImpl(AdSession(adUnitId = adUnitId, adRequest = adRequest, ad = it)) }
                notifyAdLoaded(ads)
                loadAdsFromNetworkToCache()
            }

            override fun onFailure(e: Throwable) {
                val adError = if (e is AdError) e else AdErrorPool.GENERAL_LOAD_ERROR.toAdError(e)
                notifyAdLoadFailed(adError)
                loadAdsFromNetworkToCache()
            }
        })
    }

    private fun saveAdsToCache(ads: List<Ad>) {
        CoroutineScope(Dispatchers.IO).launch {
            cache.putAds(adUnitId, ads)
        }
    }

    private fun loadAdsFromCache(maxNumberOfAds: Int, callback: Callback<List<Ad>>) {
        CoroutineScope(Dispatchers.IO).launch {
            val ads = cache.getAds(adUnitId, maxNumberOfAds)
            if (ads.isEmpty()) {
                callback.onFailure(AdErrorPool.GENERAL_LOAD_ERROR.toAdError())
            } else {
                callback.onSuccess(ads)
            }
        }
    }

    private fun loadAdsFromNetwork(adRequest: AdRequest, maxNumberOfAds: Int, callback: Callback<List<Ad>>) {
        val task = GetAdsTask2(
            context,
            AdFormat.APP_OPEN.name.lowercase(),
            adUnitId,
            adRequest,
            maxNumberOfAds,
        )
        task.getAds(callback)
    }

    private fun notifyAdLoaded(ads: List<AppOpenAd>) {
        CoroutineScope(Dispatchers.Main).launch {
            listener.onAdLoaded(ads)
        }
    }

    private fun notifyAdLoadFailed(adError: AdError) {
        CoroutineScope(Dispatchers.Main).launch {
            listener.onAdLoadFailed(adError)
        }
    }
}
