package com.particles.android.ads.internal.rendering

import android.app.Activity
import android.content.Context
import android.graphics.drawable.Drawable
import android.util.AttributeSet
import android.view.Gravity
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams.WRAP_CONTENT
import android.widget.ImageView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestListener
import com.particles.android.ads.internal.NativeAdImpl

class MediaViewImageRenderer @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : MediaViewRenderer(context, attrs, defStyleAttr) {

    private val imageView: ImageView = ImageView(context)

    init {
        super.addView(imageView, LayoutParams(WRAP_CONTENT, WRAP_CONTENT, Gravity.CENTER))
    }

    fun setImageScaleType(scaleType: ImageView.ScaleType) {
        imageView.scaleType = scaleType
    }

    override fun setNativeAd(ad: NativeAdImpl?) {
        imageView.isClickable = !(ad?.creative?.isImageClickable ?: false)

        val uri = if (ad != null && ad.images.isNotEmpty()) {
            ad.images[0].uri
        } else null

        setImageUri(uri)
    }

    private fun setImageUri(uri: String?, listener: RequestListener<Drawable>? = null) {
        if (isValidContextForGlide(context)) {
            Glide.with(context).load(uri).addListener(listener).into(imageView)
        }
    }

    private fun isValidContextForGlide(context: Context?): Boolean {
        if (context == null) {
            return false
        }

        if (context is Activity) {
            if (context.isDestroyed || context.isFinishing) {
                return false
            }
        }

        return true
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        imageView.layoutParams?.let {
            it.width = when (MeasureSpec.getMode(widthMeasureSpec)) {
                MeasureSpec.EXACTLY -> ViewGroup.LayoutParams.MATCH_PARENT
                else -> WRAP_CONTENT
            }
            it.height = when (MeasureSpec.getMode(heightMeasureSpec)) {
                MeasureSpec.EXACTLY -> ViewGroup.LayoutParams.MATCH_PARENT
                else -> WRAP_CONTENT
            }
        }
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)
    }
}
