package com.particles.android.ads.internal.rendering.video

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import android.widget.TextView
import androidx.media3.common.C
import androidx.media3.common.Player
import androidx.media3.common.util.Util
import com.particles.android.ads.R
import java.util.*

class VideoPlayerController @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    interface UserInteractionListener {
        fun onUserPlay()
        fun onUserPause()
        fun onUserMute()
        fun onUserUnmute()
    }

    interface ProgressUpdateListener {
        fun onDurationUpdate(duration: Long)
        fun onProgressUpdate(position: Long, bufferedPosition: Long)
    }

    companion object {
        const val MAX_UPDATE_INTERVAL_MS = 1000L
        const val DEFAULT_TIME_BAR_MIN_UPDATE_INTERVAL_MS = 200L
    }

    private val componentListener = ComponentListener()
    private val playButton: View?
    private val pauseButton: View?
    private val volumeOnButton: View?
    private val volumeOffButton: View?
    private val remainingDurationView: TextView?
    private val formatBuilder = StringBuilder()
    private val formatter: Formatter = Formatter(formatBuilder, Locale.getDefault())
    private var showRemainingDurationView = false

    private val updateProgressAction = this::updateProgress

    private var player: Player? = null
    private var progressUpdateListener: ProgressUpdateListener? = null
    private var userInteractionListener: UserInteractionListener? = null

    private var currentWindowOffset: Long = 0
    private var currentPosition: Long = C.TIME_UNSET
    private var currentBufferedPosition: Long = C.TIME_UNSET
    private var duration: Long = C.TIME_UNSET

    init {
        LayoutInflater.from(context).inflate(R.layout._nova_native_media_video_controller, this)

        playButton = findViewById(R.id.player_controller_play)
        playButton?.setOnClickListener(componentListener)
        pauseButton = findViewById(R.id.player_controller_pause)
        pauseButton?.setOnClickListener(componentListener)
        volumeOnButton = findViewById(R.id.player_controller_volume_on)
        volumeOnButton?.setOnClickListener(componentListener)
        volumeOffButton = findViewById(R.id.player_controller_volume_off)
        volumeOffButton?.setOnClickListener(componentListener)
        remainingDurationView = findViewById(R.id.player_controller_remaining_duration)
        remainingDurationView?.visibility = if (showRemainingDurationView) VISIBLE else GONE
    }

    fun setPlayer(player: Player?) {
        if (this.player == player) {
            return
        }
        this.player?.removeListener(componentListener)
        this.player = player
        this.player?.addListener(componentListener)
        updateAll()
    }

    fun setProgressUpdateListener(listener: ProgressUpdateListener?) {
        progressUpdateListener = listener
    }

    fun setUserInteractionListener(listener: UserInteractionListener?) {
        userInteractionListener = listener
    }

    fun show() {
        if (!isVisible()) {
            visibility = VISIBLE
            updateAll()
        }
    }

    fun hide() {
        if (isVisible()) {
            visibility = GONE
            removeCallbacks(updateProgressAction)
        }
    }

    fun isVisible(): Boolean {
        return visibility == VISIBLE
    }

    private fun updateAll() {
        updatePlayPauseButton()
        updateVolumeButton()
        updateTimeline()
    }

    private fun updatePlayPauseButton() {
        if (!isVisible() || !isAttachedToWindow) {
            return
        }

        val shouldShowPauseButton = shouldShowPauseButton()
        playButton?.visibility = if (shouldShowPauseButton) GONE else VISIBLE
        pauseButton?.visibility = if (shouldShowPauseButton) VISIBLE else GONE
    }

    private fun updateVolumeButton() {
        if (!isVisible() || !isAttachedToWindow) {
            return
        }

        val volume = player?.volume ?: 0F
        volumeOnButton?.visibility = if (volume > 0F) VISIBLE else GONE
        volumeOffButton?.visibility = if (volume > 0F) GONE else VISIBLE
    }

    private fun updateTimeline() {
        val player = this.player ?: return

        currentWindowOffset = 0
        val duration = player.contentDuration
        val durationChanged = duration != this.duration
        this.duration = duration
        if (durationChanged) {
            progressUpdateListener?.onDurationUpdate(duration)
        }

        updateProgress()
    }

    @androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
    private fun updateProgress() {
        if (!isVisible() || !isAttachedToWindow) {
            return
        }

        val player = this.player

        var position = 0L
        var bufferedPosition = 0L
        if (player != null) {
            position = currentWindowOffset + player.contentPosition
            bufferedPosition = currentWindowOffset + player.contentBufferedPosition
        }
        val positionChanged = position != currentPosition
        val bufferedPositionChanged = bufferedPosition != currentBufferedPosition
        currentPosition = position
        currentBufferedPosition = bufferedPosition

        if (remainingDurationView != null && positionChanged) {
            val remainingDuration = (duration - position).coerceAtLeast(0L)
            remainingDurationView.text =
                Util.getStringForTime(formatBuilder, formatter, remainingDuration)
            val shouldShow = position <= 5000L && duration > 0L
            setShowRemainingDurationView(shouldShow)
        }

        if (positionChanged || bufferedPositionChanged) {
            progressUpdateListener?.onProgressUpdate(position, bufferedPosition)
        }

        // Cancel any pending updates and schedule a new one if necessary.
        removeCallbacks(updateProgressAction)
        val playbackState = player?.playbackState ?: Player.STATE_IDLE
        if (player != null && player.isPlaying) {
            var mediaTimeDelayMs = MAX_UPDATE_INTERVAL_MS

            // Limit delay to the start of the next full second to ensure position display is smooth.
            val mediaTimeUntilNextFullSecondMs = 1000 - position % 1000
            mediaTimeDelayMs = mediaTimeDelayMs.coerceAtMost(mediaTimeUntilNextFullSecondMs)

            // Calculate the delay until the next update in real time, taking playback speed into account.
            val playbackSpeed = player.playbackParameters.speed
            var delayMs = if (playbackSpeed > 0) (mediaTimeDelayMs / playbackSpeed).toLong() else MAX_UPDATE_INTERVAL_MS

            // Constrain the delay to avoid too frequent / infrequent updates.
            delayMs = delayMs.coerceIn(
                DEFAULT_TIME_BAR_MIN_UPDATE_INTERVAL_MS,
                MAX_UPDATE_INTERVAL_MS
            )
            postDelayed(updateProgressAction, delayMs)
        } else if (playbackState != Player.STATE_ENDED && playbackState != Player.STATE_IDLE) {
            postDelayed(updateProgressAction, MAX_UPDATE_INTERVAL_MS)
        }
    }

    private fun setShowRemainingDurationView(show: Boolean) {
        if (showRemainingDurationView != show) {
            showRemainingDurationView = show
            val view = remainingDurationView ?: return
            view.animate()
                .alpha(if (show) 1F else 0F)
                .setDuration(250)
                .setListener(object: AnimatorListenerAdapter() {
                    override fun onAnimationStart(animation: Animator) {
                        if (show) {
                            view.visibility = VISIBLE
                        }
                    }

                    override fun onAnimationEnd(animation: Animator) {
                        if (!show) {
                            view.visibility = GONE
                        }
                    }
                })
                .start()
        }
    }

    private fun shouldShowPauseButton(): Boolean {
        return player != null && player!!.playbackState != Player.STATE_ENDED && player!!.playbackState != Player.STATE_IDLE && player!!.playWhenReady
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        updateAll()
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        removeCallbacks(updateProgressAction)
    }

    private fun dispatchPlay(player: Player) {
        val state = player.playbackState
        if (state == Player.STATE_IDLE) {
            player.prepare()
        } else if (state == Player.STATE_ENDED) {
            player.seekTo(player.currentMediaItemIndex, C.TIME_UNSET)
        }
        player.play()
    }

    private fun dispatchPause(player: Player) {
        player.pause()
    }

    private fun dispatchVolume(player: Player, volume: Float) {
        player.volume = volume
    }

    private inner class ComponentListener : Player.Listener, OnClickListener {
        override fun onEvents(player: Player, events: Player.Events) {
            if (events.containsAny(
                    Player.EVENT_PLAYBACK_STATE_CHANGED,
                    Player.EVENT_PLAY_WHEN_READY_CHANGED
                )
            ) {
                updatePlayPauseButton()
            }

            if (events.containsAny(
                    Player.EVENT_PLAYBACK_STATE_CHANGED,
                    Player.EVENT_PLAY_WHEN_READY_CHANGED,
                    Player.EVENT_IS_PLAYING_CHANGED
                )
            ) {
                updateProgress()
            }

            if (events.containsAny(
                    Player.EVENT_POSITION_DISCONTINUITY,
                    Player.EVENT_TIMELINE_CHANGED
                )
            ) {
                updateTimeline()
            }

            if (events.containsAny(Player.EVENT_VOLUME_CHANGED)) {
                updateVolumeButton()
            }
        }

        override fun onClick(v: View?) {
            val player = player ?: return
            when (v) {
                playButton -> {
                    dispatchPlay(player)
                    userInteractionListener?.onUserPlay()
                }
                pauseButton -> {
                    dispatchPause(player)
                    userInteractionListener?.onUserPause()
                }
                volumeOnButton -> {
                    dispatchVolume(player, 0F)
                    userInteractionListener?.onUserMute()
                }
                volumeOffButton -> {
                    dispatchVolume(player, 1F)
                    userInteractionListener?.onUserUnmute()
                }
            }
        }
    }
}
