package com.particles.android.ads.internal.util

import android.content.Context
import android.location.Geocoder
import android.location.Location
import android.net.ConnectivityManager
import android.os.Build
import android.provider.Settings.Secure
import android.telephony.TelephonyManager
import java.io.IOException
import java.lang.reflect.InvocationTargetException
import java.util.*

class DeviceInfo(val context: Context) {

    private val advertisingInfo by lazy { getAdvertisingInfo(context) }
    val advertisingId: String? get() = advertisingInfo.first
    val limitAdTrackingEnabled: Boolean get() = advertisingInfo.second
    val appVersion by lazy { getAppVersionName(context) }
    val osName = "android"
    val osVersion: String? = Build.VERSION.RELEASE
    val brand: String? = Build.BRAND
    val manufacturer: String? = Build.MANUFACTURER
    val model: String? = Build.MODEL
    val carrier by lazy { getCarrier(context) }
    val connectionType by lazy { getConnectionType(context) }
    val country by lazy { getCountry(context) }
    val language by lazy { getLanguage() }

    companion object {

        fun generateUUID(): String {
            return UUID.randomUUID().toString()
        }

        fun getMostRecentLocation(context: Context): Location? {
            return null // TODO: will implement later
        }

        fun getAppVersionName(context: Context): String? {
            return try {
                context.packageManager.getPackageInfo(context.packageName, 0).versionName
            } catch (e: Exception) {
                null
            }
        }

        private fun getCarrier(context: Context): String? {
            return try {
                val tm = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
                tm.networkOperatorName
            } catch (e: Exception) {
                null
            }
        }

        private fun getConnectionType(context: Context): String {
            try {
                val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
                val activeNetworkInfo = cm.activeNetworkInfo
                if (activeNetworkInfo == null || !activeNetworkInfo.isAvailable || !activeNetworkInfo.isConnected) {
                    return "UNKNOWN"
                }
                return when (activeNetworkInfo.type) {
                    ConnectivityManager.TYPE_ETHERNET -> "ETH"
                    ConnectivityManager.TYPE_WIFI -> "WIFI"
                    ConnectivityManager.TYPE_WIMAX -> "4G" // Regarded as 4G
                    ConnectivityManager.TYPE_MOBILE,
                    ConnectivityManager.TYPE_MOBILE_MMS,
                    ConnectivityManager.TYPE_MOBILE_SUPL,
                    ConnectivityManager.TYPE_MOBILE_DUN,
                    ConnectivityManager.TYPE_MOBILE_HIPRI -> "${getMobileNetworkGeneration(activeNetworkInfo.subtype)}G"
                    else -> "UNKNOWN"
                }
            } catch (e: Exception) {
                return "UNKNOWN"
            }
        }

        private fun getMobileNetworkGeneration(type: Int): Int {
            return when (type) {
                TelephonyManager.NETWORK_TYPE_GPRS,
                TelephonyManager.NETWORK_TYPE_EDGE,
                TelephonyManager.NETWORK_TYPE_CDMA,
                TelephonyManager.NETWORK_TYPE_1xRTT,
                TelephonyManager.NETWORK_TYPE_IDEN,
                TelephonyManager.NETWORK_TYPE_GSM -> 2
                TelephonyManager.NETWORK_TYPE_UMTS,
                TelephonyManager.NETWORK_TYPE_EVDO_0,
                TelephonyManager.NETWORK_TYPE_EVDO_A,
                TelephonyManager.NETWORK_TYPE_HSDPA,
                TelephonyManager.NETWORK_TYPE_HSUPA,
                TelephonyManager.NETWORK_TYPE_HSPA,
                TelephonyManager.NETWORK_TYPE_EVDO_B,
                TelephonyManager.NETWORK_TYPE_EHRPD,
                TelephonyManager.NETWORK_TYPE_HSPAP,
                TelephonyManager.NETWORK_TYPE_TD_SCDMA -> 3
                TelephonyManager.NETWORK_TYPE_LTE,
                TelephonyManager.NETWORK_TYPE_IWLAN -> 4
                TelephonyManager.NETWORK_TYPE_NR -> 5
                else -> 0
            }
        }

        private fun getCountry(context: Context): String? {
            // This should not be called on the main thread.

            // Prioritize reverse geocode, but until we have a result from that,
            // we try to grab the country from the network, and finally the locale
            var country = getCountryFromLocation(context)
            if (!country.isNullOrEmpty()) {
                return country
            }

            country = getCountryFromNetwork(context)
            if (!country.isNullOrEmpty()) {
                return country
            }

            return Locale.getDefault().country
        }

        private fun getCountryFromLocation(context: Context): String? {
            val recent: Location? = getMostRecentLocation(context)
            if (recent != null) {
                try {
                    if (Geocoder.isPresent()) {
                        val geocoder = Geocoder(context, Locale.ENGLISH)
                        val addresses = geocoder.getFromLocation(recent.latitude, recent.longitude, 1)
                        if (addresses != null) {
                            for (address in addresses) {
                                if (address != null) {
                                    return address.countryCode
                                }
                            }
                        }
                    }
                } catch (e: IOException) {
                    // Failed to reverse geocode location
                } catch (e: NullPointerException) {
                    // Failed to reverse geocode location
                } catch (e: NoSuchMethodError) {
                    // failed to fetch geocoder
                } catch (e: IllegalArgumentException) {
                    // Bad lat / lon values can cause Geocoder to throw IllegalArgumentExceptions
                } catch (e: IllegalStateException) {
                    // sometimes the location manager is unavailable
                }
            }
            return null
        }

        private fun getCountryFromNetwork(context: Context): String? {
            try {
                val tm = context
                    .getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
                if (tm.phoneType != TelephonyManager.PHONE_TYPE_CDMA) {
                    val country = tm.networkCountryIso
                    if (country != null) {
                        return country.uppercase(Locale.US)
                    }
                }
            } catch (e: Exception) {
                // Failed to get country from network
            }
            return null
        }

        private fun getLanguage(): String? {
            return Locale.getDefault().language
        }

        private fun getAdvertisingInfo(context: Context): Pair<String?, Boolean> {
            // This should not be called on the main thread.
            return if ("Amazon" == Build.MANUFACTURER) {
                getAmazonAdvertisingInfo(context)
            } else {
                getGoogleAdvertisingInfo(context)
            }
        }

        private fun getAmazonAdvertisingInfo(context: Context): Pair<String?, Boolean> {
            val cr = context.contentResolver
            val lat = Secure.getInt(cr, "limit_ad_tracking", 0) == 1
            val id = Secure.getString(cr, "advertising_id")
            return Pair(id, lat)
        }

        private fun getGoogleAdvertisingInfo(context: Context): Pair<String?, Boolean> {
            var id: String? = null
            var lat = false
            try {
                val advertisingIdClient = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient")
                val getAdvertisingInfo = advertisingIdClient.getMethod("getAdvertisingIdInfo", Context::class.java)
                val advertisingInfo = getAdvertisingInfo.invoke(null, context)
                val isLimitAdTrackingEnabled = advertisingInfo.javaClass.getMethod("isLimitAdTrackingEnabled")

                val limitAdTrackingEnabled = isLimitAdTrackingEnabled.invoke(advertisingInfo) as Boolean?
                lat = limitAdTrackingEnabled != null && limitAdTrackingEnabled

                val getId = advertisingInfo.javaClass.getMethod("getId")
                id = getId.invoke(advertisingInfo) as String
            } catch (e: ClassNotFoundException) {
                // Google Play Services SDK not found!
            } catch (e: InvocationTargetException) {
                // Google Play Services not available
            } catch (e: Exception) {
                // Encountered an error connecting to Google Play Services
            }
            return Pair(id, lat)
        }
    }
}
