package com.particles.android.ads.browser

import com.particles.android.ads.internal.domain.BrowserExtras
import com.particles.android.ads.internal.domain.TrackingEvent
import com.particles.android.ads.internal.tracking.EventTracker

class BrowserTracker(
    private val adUnitId: String,
    private val adClickedAt: Long,
    private val encryptedAdToken: String,
) {
    companion object {
        const val STATUS_IDLE = 0
        const val STATUS_STARTED = 1
        const val STATUS_TEXT_LOADED = 2
        const val STATUS_ALL_LOADED = 3
        const val STATUS_CLOSED = 4
    }

    private var status = STATUS_IDLE
    private var resumedSeq = 0
    private var pageIndex = 0
    private var scrollDepth = 0F

    fun logPageStart() {
        if (status >= STATUS_STARTED) {
            return
        }

        status = STATUS_STARTED
        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_START,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
            )
        )
    }

    fun logTextLoaded() {
        if (status >= STATUS_TEXT_LOADED) {
            return
        }

        status = STATUS_TEXT_LOADED
        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_TEXT_LOADED,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
            )
        )
    }

    fun logPageLoaded() {
        if (status >= STATUS_ALL_LOADED) {
            return
        }

        if (status < STATUS_TEXT_LOADED) {
            logTextLoaded()
        }

        status = STATUS_ALL_LOADED
        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_LOADED,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
            )
        )
    }

    fun logPageClose(reason: String, pageIndex: Int, scrollDepth: Float) {
        if (status >= STATUS_CLOSED) {
            return
        }

        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_CLOSE,
                reason = reason,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
                browserExtras = BrowserExtras(
                    status = status,
                    pageIndex = pageIndex,
                    scrollDepth = scrollDepth,
                ),
            )
        )
        status = STATUS_CLOSED
    }

    fun logPagePause(pageIndex: Int, scrollDepth: Float) {
        if (status >= STATUS_CLOSED) {
            return
        }

        this.pageIndex = pageIndex
        this.scrollDepth = scrollDepth
        this.resumedSeq += 1
        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_PAUSE,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
                browserExtras = BrowserExtras(
                    seq = resumedSeq,
                    status = status,
                    pageIndex = pageIndex,
                    scrollDepth = scrollDepth,
                ),
            )
        )
    }

    fun logPageResume() {
        if (status >= STATUS_CLOSED) {
            return
        }

        // ignore the first resume event since we already have a start event
        if (resumedSeq <= 0) {
            return
        }

        val duration = System.currentTimeMillis() - adClickedAt

        EventTracker.fire(
            TrackingEvent(
                type = TrackingEvent.EVENT_TYPE_BROWSER_RESUME,
                adUnitId = adUnitId,
                encryptedAdToken = encryptedAdToken,
                duration = duration,
                browserExtras = BrowserExtras(
                    seq = resumedSeq,
                    status = status,
                    pageIndex = pageIndex,
                    scrollDepth = scrollDepth,
                ),
            )
        )
    }
}
