package com.particles.android.ads.internal

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.view.View
import androidx.browser.customtabs.CustomTabsIntent
import com.particles.android.ads.browser.BrowserActivity
import com.particles.android.ads.browser.BrowserMediaHeader
import com.particles.android.ads.internal.domain.BrowserOption
import com.particles.android.ads.internal.domain.CampaignObjective
import com.particles.android.ads.nativead.MediaView
import com.particles.android.ads.nativead.NativeAd

class AdClickHandler(
    private val view: View,
    private val context: Context = view.context,
) {

    fun handleClick(ad: BaseAdImpl, url: String? = null) {
        val adSession = ad.adSession
        val creative = adSession.ad.creative
        val clickThroughUrl = if (!url.isNullOrEmpty()) url else creative.clickThroughUrl
        if (clickThroughUrl.isEmpty()) {
            return
        }

        when (creative.browserOption) {
            BrowserOption.IN_APP -> openAdInWebView(
                context,
                clickThroughUrl,
                ad,
            )

            BrowserOption.BROWSER -> try {
                openAdInBrowser(context, clickThroughUrl)
            } catch (e: ActivityNotFoundException) {
                // ignore
            }
        }
    }

    private fun openAdInBrowser(context: Context, url: String) {
        val builder = CustomTabsIntent.Builder()
        val customTabsIntent = builder.build()
        if (context !is Activity) {
            customTabsIntent.intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        }
        customTabsIntent.launchUrl(context, Uri.parse(url))
    }

    private fun openAdInWebView(
        context: Context,
        url: String,
        ad: BaseAdImpl,
    ) {
        val intent = Intent(context, BrowserActivity::class.java)
        intent.putExtra("url", url)
        intent.putExtra("request_id", ad.requestId)
        intent.putExtra("ad_id", ad.adId)
        intent.putExtra("ad_unit_id", ad.adUnitId)
        intent.putExtra("ad_clicked_at", ad.adSession.adClickedAt)
        intent.putExtra("ad_token", ad.adSession.ad.encryptedAdToken)
        intent.putExtra("is_app_install", ad.adSession.ad.creative.campaignObjective == CampaignObjective.APP_CONVERSION)
        intent.putExtra("ad_bundle_id", ad.adSession.ad.creative.adBundleId)
        if (ad is NativeAdImpl && view is MediaView
            && ad.creativeType == NativeAd.CreativeType.VIDEO
            && ad.creative.video?.isContinuePlay == true
        ) {
            BrowserMediaHeader.putAdToIntent(ad, intent)
        }
        if (context !is Activity) {
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        }
        context.startActivity(intent)
    }
}
