package com.particles.android.ads.internal.data.mapper

import android.widget.ImageView
import com.particles.android.ads.internal.domain.AbConfig
import com.particles.android.ads.internal.data.entity.AdEntity
import com.particles.android.ads.internal.data.entity.AddonItemEntity
import com.particles.android.ads.internal.data.entity.CarouselItemEntity
import com.particles.android.ads.internal.data.entity.CreativeEntity
import com.particles.android.ads.internal.data.entity.PlayableItemEntity
import com.particles.android.ads.internal.data.entity.VideoItemEntity
import com.particles.android.ads.internal.domain.*

object AdEntityDataMapper : Mapper<AdEntity, Ad> {
    override fun map(source: AdEntity): Ad {
        return Ad(
            raw = source.raw,
            requestId = source.requestId,
            adId = source.adId,
            adSetId = source.adSetId,
            creative = CreativeEntityDataMapper.map(source.creative),
            price = source.price,
            startTimeMillis = source.startTimeMs,
            expireTimeMillis = source.expirationMs,
            encryptedAdToken = source.encryptedAdToken,
            abConfig = AbConfig(source.abConfig ?: emptyMap())
        )
    }
}

object CreativeEntityDataMapper : Mapper<CreativeEntity, Creative> {
    override fun map(source: CreativeEntity): Creative {
        return Creative(
            type = source.creativeType,
            adm = source.adm,
            headline = source.headline,
            body = source.body,
            icon = toImage(source.iconUrl),
            video = source.videoItem?.let { VideoEntityDataMapper.map(it) },
            addon = source.addonItem?.let { AddonEntityDataMapper.map(it) },
            playable = source.playableItem?.let { PlayableEntityDataMapper.map(it) },
            images = if (source.imageUrls.isNotEmpty()) {
                source.imageUrls.mapNotNull { toImage(it) }
            } else {
                listOfNotNull(toImage(source.imageUrl))
            },
            imageScaleType = when (source.imageScaleMode) {
                "crop" -> ImageView.ScaleType.CENTER_CROP
                "fit" -> ImageView.ScaleType.FIT_CENTER
                else -> null
            },
            isImageClickable = source.isImageClickable,
            mediaLayoutType = source.layout.takeIf { it.isNotEmpty() }
                ?: when (source.creativeType) {
                    "IMAGE" -> source.isVerticalImage
                    "VIDEO" -> source.videoItem?.isVertical == true
                    else -> false
                }.let { if (it) "vertical" else "horizontal" },
            starRating = 0.0,
            advertiser = source.advertiser,
            callToAction = source.callToAction,
            clickThroughUrl = source.ctrUrl,
            browserOption = toBrowserOption(source.launchOption),
            campaignObjective = toCampaignObjective(source.campaignObjective),
            adBundleId = source.adBundleId,
            carouselItems = source.carouselItems.map { CarouselEntityDataMapper.map(it) },
            impressionTrackingUrls = source.thirdPartyImpressionTrackingUrls,
            clickTrackingUrls = source.thirdPartyClickTrackingUrls,
            adVerifications = source.thirdPartyViewTrackingUrls.map {
                AdVerification(javascriptResourceUrl = it)
            },
        )
    }

    private fun toBrowserOption(option: String): BrowserOption {
        return when (option) {
            "LAUNCH_WEBVIEW" -> BrowserOption.IN_APP
            "LAUNCH_BROWSER" -> BrowserOption.BROWSER
            else -> BrowserOption.BROWSER
        }
    }

    private fun toCampaignObjective(objective: String): CampaignObjective {
        return when (objective) {
            "APP_CONVERSION" -> CampaignObjective.APP_CONVERSION
            else -> CampaignObjective.OTHER
        }
    }
}

object CarouselEntityDataMapper : Mapper<CarouselItemEntity, CarouselItem> {
    override fun map(source: CarouselItemEntity): CarouselItem {
        return CarouselItem(
            body = source.body,
            image = toImage(source.imageUrl),
            callToAction = source.callToAction,
            clickThroughUrl = source.ctrUrl,
        )
    }
}

object VideoEntityDataMapper : Mapper<VideoItemEntity, Video> {
    override fun map(source: VideoItemEntity): Video {
        return Video(
            videoUrl = source.videoUrl,
            coverUrl = source.coverUrl,
            isAutoPlay = source.isPlayAutomatically,
            isLoopPlay = source.isLoop,
            isMutePlay = source.isMute,
            isClickable = source.isVideoClickable,
            isContinuePlay = source.isPlayOnLandingPage,
        )
    }
}

object AddonEntityDataMapper : Mapper<AddonItemEntity, Addon?> {
    override fun map(source: AddonItemEntity): Addon? {
        return when (source.type) {
            "display_card" -> DisplayCardAddon(
                imageUrl = source.imageUrl,
                displayTime = source.displayTime,
            )

            else -> null
        }
    }
}

object PlayableEntityDataMapper : Mapper<PlayableItemEntity, Playable> {
    override fun map(source: PlayableItemEntity): Playable {
        return Playable(
            url = source.url,
            clickAreaMode = try {
                PlayableClickAreaMode.valueOf(source.clickAreaMode)
            } catch (e: IllegalArgumentException) {
                PlayableClickAreaMode.MEDIA
            }
        )
    }
}

private fun toImage(url: String?): Image? {
    return if (url.isNullOrEmpty()) null else Image(url, 0, 0)
}
