package com.particles.android.ads.internal.rendering

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.util.AttributeSet
import android.view.LayoutInflater
import com.particles.android.ads.R
import com.particles.android.ads.internal.NativeAdImpl
import com.particles.android.ads.internal.domain.VideoEvents
import com.particles.android.ads.internal.rendering.video.VideoPlayerView
import com.particles.android.ads.internal.util.Stopwatch
import com.particles.android.ads.internal.util.VisibilityTracker
import com.particles.android.ads.internal.util.viewability.OMMediaEvents

class MediaViewVideoRenderer @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : MediaViewRenderer(context, attrs, defStyleAttr) {

    private val handler = Handler(Looper.getMainLooper())
    private val videoView: VideoPlayerView?
    private var ad: NativeAdImpl? = null
    private val autoplayHelper = AutoplayHelper()
    private val visibilityTracker = VisibilityTracker(this,
        onVisibilityChanged = this::onVisibilityChanged)

    init {
        val view = LayoutInflater.from(context).inflate(R.layout._nova_native_media_video, this)
        videoView = view.findViewById(R.id.video_player)
    }

    fun setUseController(useController: Boolean) {
        videoView?.setUseController(useController)
    }

    override fun setNativeAd(ad: NativeAdImpl?, omMediaEvents: OMMediaEvents?) {
        if (this.ad != ad) {
            this.ad = ad

            if (ad == null || ad.creative.video == null) {
                return
            }

            val video = ad.creative.video
            val state = ad.adSession.adPlaybackState

            val started = state.position >= 0

            val coverUrl = if (started) null else video.coverUrl

            val volume = if (started) state.volume else if (video.isMutePlay) 0F else 1F

            val autoplayDelayMs = if (started && state.playWhenReady)
                0L
            else if (!started && video.isAutoPlay)
                if (coverUrl.isNullOrEmpty()) 0L else 1000L
            else
                -1L

            videoView?.isClickable = !video.isClickable
            videoView?.setListener(VideoEventListener(ad, VideoEvents(ad.adSession, omMediaEvents)))
            videoView?.setDataSource(
                video.videoUrl,
                coverUrl = coverUrl,
                position = state.position,
                volume = volume,
                looping = video.isLoopPlay
            )
            autoplayHelper.playDelayed(autoplayDelayMs)
            if (ad.creative.video.isContinuePlay) {
                ad.adSession.adPlaybackState.playWhenReady = autoplayDelayMs >= 0
            }
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        visibilityTracker.startTracking()
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        if (ad?.creative?.video?.isContinuePlay != true) {
            savePlaybackState()
        }
        visibilityTracker.stopTracking()
        videoView?.release()
        this.ad = null
    }

    private var visiblePixels = 0L

    private fun onVisibilityChanged(visiblePixels: Long, totalPixels: Long) {
        if (this.visiblePixels == 0L && visiblePixels > 0 && ad?.creative?.video?.isContinuePlay == true) {
            ad?.adSession
                ?.takeIf { it.adClickedAt > 0 && it.adPlaybackState.position > 0 }
                ?.let {
                    videoView?.update(
                        it.adPlaybackState.position,
                        it.adPlaybackState.volume,
                        it.adPlaybackState.playWhenReady,
                    )
                }
        }
        this.visiblePixels = visiblePixels
        val visiblePercent = if (totalPixels > 0) visiblePixels * 100 / totalPixels else 0
        autoplayHelper.setPlayable(visiblePercent > 95)
    }

    private fun savePlaybackState() {
        videoView?.getPlayerState()?.let {
            val output = ad?.adSession?.adPlaybackState
            output?.position = it.position
            output?.volume = it.volume
            output?.playWhenReady = it.playWhenReady || autoplayHelper.needAutoPlay()
        }
    }

    private inner class AutoplayHelper {
        private var playable = false
        private var playDelayMills = -1L
        private val playRunnable = Runnable { videoView?.resume(VideoPlayerView.REASON_VISIBILITY_CHANGED) }

        fun playDelayed(delayMills: Long) {
            handler.removeCallbacks(playRunnable)
            playDelayMills = delayMills
            if (playable) {
                schedulePlay()
            }
        }

        fun setPlayable(playable: Boolean) {
            if (this.playable != playable) {
                this.playable = playable
                onPlayableChanged()
            }
        }

        fun needAutoPlay(): Boolean {
            return playDelayMills >= 0
        }

        private fun onPlayableChanged() {
            handler.removeCallbacks(playRunnable)
            when (playable) {
                true -> schedulePlay()
                false -> if (videoView?.isPlaying() == true) {
                    videoView.pause(VideoPlayerView.REASON_VISIBILITY_CHANGED)
                    playDelayMills = 0
                }
            }
        }

        private fun schedulePlay() {
            if (playDelayMills >= 0) {
                handler.postDelayed(playRunnable, playDelayMills)
                playDelayMills = -1
            }
        }
    }

    private class VideoEventListener(private val ad: NativeAdImpl, private val events: VideoEvents) : VideoPlayerView.Listener {
        private var position = 0L
        private var duration = 0L
        private var percent = -1
        private var isPlayingOrBuffering = false
        private var stopwatch = Stopwatch()

        override fun onIsPlayingOrBufferingChanged(isPlayingOrBuffering: Boolean) {
            this.isPlayingOrBuffering = isPlayingOrBuffering
            if (isPlayingOrBuffering) {
                stopwatch.start()
            } else {
                stopwatch.stop()
            }
        }

        override fun onPlay(reason: String) {
            events.onVideoResume(reason)
            if (ad.creative.video?.isContinuePlay == true && reason == VideoPlayerView.REASON_USER_INTERACTION) {
                ad.adSession.adPlaybackState.playWhenReady = true
            }
        }

        override fun onPause(reason: String) {
            events.onVideoPause(reason, duration.toInt(), position.toInt())
            if (ad.creative.video?.isContinuePlay == true && reason == VideoPlayerView.REASON_USER_INTERACTION) {
                ad.adSession.adPlaybackState.playWhenReady = false
            }
        }

        override fun onUserMute() {
            events.onVideoMute()
            if (ad.creative.video?.isContinuePlay == true) {
                ad.adSession.adPlaybackState.volume = 0F
            }
        }

        override fun onUserUnmute() {
            events.onVideoUnmute()
            if (ad.creative.video?.isContinuePlay == true) {
                ad.adSession.adPlaybackState.volume = 1F
            }
        }

        override fun onDurationUpdate(duration: Long) {
            this.duration = duration
            ad.mediaListener?.onDurationUpdate(duration)
        }

        override fun onProgressUpdate(position: Long, bufferedPosition: Long) {
            ad.mediaListener?.onProgressUpdate(position, bufferedPosition)
            val duration = duration
            if (position < 0 || duration <= 0) {
                return
            }

            // progress events by position mills
            if (this.position < 3000 && position >= 3000) {
                events.onVideoProgress(3000)
            }
            this.position = position
            if (ad.creative.video?.isContinuePlay == true) {
                ad.adSession.adPlaybackState.position = position
            }

            // progress events by percentage
            val percent = (position * 100 / duration).toInt()
            if (this.percent < 0 && percent >= 0) {
                events.onVideoStart(duration.toInt(), stopwatch.elapsedMills().toInt())
            } else if (this.percent < 25 && percent >= 25) {
                events.onVideoFirstQuartile()
            } else if (this.percent < 50 && percent >= 50) {
                events.onVideoMidpoint()
            } else if (this.percent < 75 && percent >= 75) {
                events.onVideoThirdQuartile()
            } else if (this.percent < 100 && percent >= 100) {
                stopwatch.reset()
                if (isPlayingOrBuffering) {
                    stopwatch.start()
                }
                events.onVideoComplete()
            }
            if (this.percent >= 100 && percent < 100) {
                ad.adSession.adPlaybackState.repeatCount++
            }
            this.percent = percent
        }

    }
}
