package com.particles.android.ads.internal.rendering.video

import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import com.bumptech.glide.Glide
import androidx.media3.common.C
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.common.AudioAttributes
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory
import androidx.media3.exoplayer.source.MediaSource
import androidx.media3.ui.PlayerView
import androidx.media3.datasource.DefaultHttpDataSource
import androidx.media3.datasource.cache.CacheDataSink
import androidx.media3.datasource.cache.CacheDataSource
import com.particles.android.ads.R

class VideoPlayerView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    companion object {
        const val REASON_USER_INTERACTION = "manual"
        const val REASON_VISIBILITY_CHANGED = "auto"
    }

    interface Listener : VideoPlayerController.UserInteractionListener, VideoPlayerController.ProgressUpdateListener {
        fun onIsPlayingOrBufferingChanged(isPlayingOrBuffering: Boolean)
        fun onPlay(reason: String)
        fun onPause(reason: String)

        override fun onUserPlay() {
            onPlay(REASON_USER_INTERACTION)
        }

        override fun onUserPause() {
            onPause(REASON_USER_INTERACTION)
        }
    }

    private val componentListener: ComponentListener = ComponentListener()
    private val playerView: PlayerView?
    private val bufferingView: View?
    private val controller: VideoPlayerController?
    private val coverImageView: ImageView?
    private val centerPlayButtonContainer: View?
    private val centerPlayButton: View?
    private val bigPlayButtonContainer: View?
    private val bigPlayButton: View?

    private var useController = true
    private var player: ExoPlayer? = null
    private var listener: Listener? = null

    init {
        LayoutInflater.from(context).inflate(R.layout._nova_native_media_video_player_view, this)

        playerView = findViewById(R.id.player_view)
        bufferingView = findViewById(R.id.buffering)
        bufferingView?.visibility = GONE
        controller = findViewById(R.id.player_controller)
        controller?.setProgressUpdateListener(componentListener)
        controller?.setUserInteractionListener(componentListener)
        controller?.hide()
        centerPlayButtonContainer = findViewById(R.id.player_center_play_button_container)
        centerPlayButtonContainer?.visibility = GONE
        centerPlayButton = findViewById(R.id.player_center_play_button)
        coverImageView = findViewById(R.id.player_cover_image)
        bigPlayButtonContainer = findViewById(R.id.player_big_play_button_container)
        bigPlayButtonContainer?.setOnClickListener(componentListener)
        bigPlayButton = findViewById(R.id.player_big_play_button)
        bigPlayButton?.setOnClickListener(componentListener)
        setOnClickListener(componentListener)
    }

    override fun setClickable(clickable: Boolean) {
        super.setClickable(clickable)
        bigPlayButton?.isClickable = clickable
        bigPlayButton?.isDuplicateParentStateEnabled = !clickable
        bigPlayButtonContainer?.isClickable = !clickable
    }

    fun setListener(listener: Listener) {
        this.listener = listener
    }

    fun setUseController(useController: Boolean) {
        this.useController = useController
        if (!useController) {
            controller?.hide()
        }
    }

    @androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
    fun setDataSource(
        videoUrl: String,
        coverUrl: String? = null,
        position: Long = -1,
        volume: Float = 0F,
        autoplay: Boolean = false,
        looping: Boolean = false,
    ) {
        val player = this.player?.also {
            it.stop()
            it.clearMediaItems()
        } ?: createPlayer()

        setPlayer(player)

        setCoverImageUri(coverUrl)
        if (coverUrl != null) {
            showCoverImage()
        } else {
            hideCoverImage()
        }

        player.setAudioAttributes(AudioAttributes.DEFAULT, volume > 0)
        player.setMediaSource(createMediaSource(videoUrl))
        player.volume = volume
        player.repeatMode = if (looping) {
            Player.REPEAT_MODE_ONE
        } else {
            Player.REPEAT_MODE_OFF
        }
        player.prepare()
        if (position >= 0) {
            player.seekTo(position)
        } else {
            controller?.hide()
            showBigPlayButton()
        }

        if (autoplay) {
            player.play()
        }
    }

    fun update(
        position: Long = -1,
        volume: Float = 0F,
        playWhenReady: Boolean = false,
    ) {
        if (position >= 0) {
            player?.seekTo(position)
            player?.volume = volume
            player?.playWhenReady = playWhenReady
        }
    }

    fun resume(reason: String) {
        player?.play()
        listener?.onPlay(reason)
    }

    fun isPlaying(): Boolean {
        return player?.playWhenReady ?: false
    }

    fun pause(reason: String) {
        player?.pause()
        listener?.onPause(reason)
    }

    fun release() {
        val player = this.player
        setPlayer(null)
        player?.release()
    }

    fun getPlayerState(): PlayerState? {
        if (coverImageView?.visibility == View.VISIBLE) return null
        return player?.let {
            PlayerState(
                it.currentPosition,
                it.volume,
                it.playWhenReady,
            )
        }
    }

    private fun setPlayer(player: ExoPlayer?) {
        if (this.player == player) {
            return
        }
        this.player?.removeListener(componentListener)
        this.player = player
        this.playerView?.player = player
        this.controller?.setPlayer(player)
        updateBuffering()
        this.player?.addListener(componentListener)
    }

    private fun createPlayer(): ExoPlayer {
        return ExoPlayer.Builder(context)
            .build()
    }

    @androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
    private fun createMediaSource(uri: String): MediaSource {
        val cache = NovaVideoCacheManager.getSimpleCache(context)
        val factory = DefaultMediaSourceFactory(
            CacheDataSource.Factory()
                .setCache(cache)
                .setCacheWriteDataSinkFactory(CacheDataSink.Factory().setCache(cache))
                .setUpstreamDataSourceFactory(DefaultHttpDataSource.Factory())
                .setFlags(CacheDataSource.FLAG_IGNORE_CACHE_ON_ERROR)
        )
        return factory.createMediaSource(MediaItem.fromUri(uri))
    }

    private fun updateBuffering() {
        if (bufferingView != null) {
            val showBufferingSpinner =
                player?.playbackState == Player.STATE_BUFFERING && player?.playWhenReady == true
            bufferingView.visibility = if (showBufferingSpinner) VISIBLE else GONE
        }
    }

    private fun showBigPlayButton() {
        bigPlayButtonContainer?.visibility = View.VISIBLE
        centerPlayButtonContainer?.visibility = GONE
    }

    private fun hideBigPlayButton() {
        bigPlayButtonContainer?.visibility = View.GONE
        centerPlayButtonContainer?.visibility = VISIBLE
    }

    private fun setCoverImageUri(uri: String?) {
        coverImageView?.let {
            if (isValidContextForGlide(context)) {
                Glide.with(context).load(uri).into(it)
            }
        }
    }

    private fun showCoverImage() {
        coverImageView?.visibility = View.VISIBLE
    }

    private fun hideCoverImage() {
        coverImageView?.visibility = View.GONE
    }

    private fun isValidContextForGlide(context: Context?): Boolean {
        if (context == null) {
            return false
        }

        if (context is Activity) {
            if (context.isDestroyed || context.isFinishing) {
                return false
            }
        }

        return true
    }

    private inner class ComponentListener : Player.Listener, OnClickListener,
        VideoPlayerController.ProgressUpdateListener, VideoPlayerController.UserInteractionListener {

        private var duration: Long = 0
        private var isFirstFrameRendered = false
        private var playWhenReady = false
        private var isPlayingOrBuffering = false

        override fun onRenderedFirstFrame() {
            isFirstFrameRendered = true
            if (playWhenReady) {
                hideCoverImage()
            }
        }

        override fun onPlayWhenReadyChanged(playWhenReady: Boolean, reason: Int) {
            updateBuffering()
            updateIsPlayingOrBuffering()
            this.playWhenReady = playWhenReady
            if (playWhenReady) {
                if (useController) {
                    controller?.show()
                }
                hideBigPlayButton()
                if (isFirstFrameRendered) {
                    hideCoverImage()
                }
            }
        }

        override fun onPlaybackStateChanged(playbackState: Int) {
            updateBuffering()
            updateIsPlayingOrBuffering()
            when (playbackState) {
                Player.STATE_ENDED -> {
                    controller?.hide()
                    showBigPlayButton()
                    listener?.onProgressUpdate(duration, duration)
                }
                else -> {}
            }
        }

        private fun updateIsPlayingOrBuffering() {
            val isPlayingOrBuffering = isPlayingOrBuffering()
            if (this.isPlayingOrBuffering != isPlayingOrBuffering) {
                this.isPlayingOrBuffering = isPlayingOrBuffering
                if (isClickable && !isPlayingOrBuffering) {
                    centerPlayButton?.visibility = VISIBLE
                } else {
                    centerPlayButton?.visibility = INVISIBLE
                }
                listener?.onIsPlayingOrBufferingChanged(isPlayingOrBuffering)
            }
        }

        private fun isPlayingOrBuffering(): Boolean {
            val player = player ?: return false
            return player.playbackState != Player.STATE_ENDED
                    && player.playbackState != Player.STATE_IDLE
                    && player.playWhenReady
        }

        override fun onPositionDiscontinuity(
            oldPosition: Player.PositionInfo,
            newPosition: Player.PositionInfo,
            reason: Int
        ) {
            if (reason == Player.DISCONTINUITY_REASON_AUTO_TRANSITION) {
                listener?.onProgressUpdate(duration, duration)
            }
        }

        override fun onClick(v: View?) {
            if (v == bigPlayButton || v == bigPlayButtonContainer) {
                val player = player ?: return
                when (player.playbackState) {
                    Player.STATE_IDLE -> player.prepare()
                    Player.STATE_ENDED -> player.seekTo(player.currentMediaItemIndex, C.TIME_UNSET)
                    else -> Unit
                }
                player.play()
                if (useController) {
                    controller?.show()
                }
                hideBigPlayButton()
                listener?.onUserPlay()
            } else if (v == this@VideoPlayerView) {
                val player = player ?: return
                when (player.playWhenReady) {
                    true -> {
                        player.playWhenReady = false
                        onUserPause()
                    }

                    false -> {
                        player.playWhenReady = true
                        onUserPlay()
                    }
                }
            }
        }

        override fun onUserPlay() {
            listener?.onUserPlay()
        }

        override fun onUserPause() {
            listener?.onUserPause()
        }

        override fun onUserMute() {
            player?.setAudioAttributes(AudioAttributes.DEFAULT, false)
            listener?.onUserMute()
        }

        override fun onUserUnmute() {
            player?.setAudioAttributes(AudioAttributes.DEFAULT, true)
            listener?.onUserUnmute()
        }

        override fun onDurationUpdate(duration: Long) {
            this.duration = duration
            listener?.onDurationUpdate(duration)
        }

        override fun onProgressUpdate(position: Long, bufferedPosition: Long) {
            listener?.onProgressUpdate(position, bufferedPosition)
        }
    }
}
