package com.particles.android.ads.internal.rendering.video

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import androidx.media3.common.AudioAttributes
import androidx.media3.common.C
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.datasource.DefaultHttpDataSource
import androidx.media3.datasource.cache.CacheDataSink
import androidx.media3.datasource.cache.CacheDataSource
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory
import androidx.media3.exoplayer.source.MediaSource
import androidx.media3.ui.PlayerView
import com.particles.android.ads.R

class VideoPlayerView2 @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    companion object {
        const val REASON_USER_INTERACTION = "manual"
        const val REASON_VISIBILITY_CHANGED = "auto"
    }

    interface Listener : VideoPlayerController.UserInteractionListener, VideoPlayerController.ProgressUpdateListener {
        fun onIsPlayingOrBufferingChanged(isPlayingOrBuffering: Boolean)
        fun onPlay(reason: String)
        fun onPause(reason: String)

        override fun onUserPlay() {
            onPlay(REASON_USER_INTERACTION)
        }

        override fun onUserPause() {
            onPause(REASON_USER_INTERACTION)
        }
    }

    private val componentListener: ComponentListener = ComponentListener()
    private val playerView: PlayerView?
    private val playButton: View?
    private val pauseButton: View?
    private val volumeOnButton: View?
    private val volumeOffButton: View?

    private var player: ExoPlayer? = null
    private var listener: Listener? = null

    init {
        LayoutInflater.from(context).inflate(R.layout._nova_native_media_video_player_view2, this)
        playerView = findViewById(R.id.player_view)
        playButton = findViewById(R.id.exo_play)
        pauseButton = findViewById(R.id.exo_pause)
        volumeOnButton = findViewById(R.id.exo_volume_on)
        volumeOffButton = findViewById(R.id.exo_volume_off)
        playButton.setOnClickListener(componentListener)
        pauseButton.setOnClickListener(componentListener)
        volumeOnButton.setOnClickListener(componentListener)
        volumeOffButton.setOnClickListener(componentListener)
    }

    fun setListener(listener: Listener) {
        this.listener = listener
    }

    @androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
    fun setDataSource(
        videoUrl: String,
        position: Long = -1,
        volume: Float = 0F,
        autoplay: Boolean = false,
        looping: Boolean = false,
    ) {
        val player = this.player?.also {
            it.stop()
            it.clearMediaItems()
        } ?: createPlayer()

        setPlayer(player)

        player.setMediaSource(createMediaSource(videoUrl))
        player.volume = volume
        player.repeatMode = if (looping) {
            Player.REPEAT_MODE_ONE
        } else {
            Player.REPEAT_MODE_OFF
        }
        player.prepare()
        if (position >= 0) {
            player.seekTo(position)
        }

        if (autoplay) {
            player.play()
        }
    }

    fun resume(reason: String) {
        player?.play()
        listener?.onPlay(reason)
    }

    fun isPlaying(): Boolean {
        return player?.playWhenReady ?: false
    }

    fun pause(reason: String) {
        player?.pause()
        listener?.onPause(reason)
    }

    fun release() {
        val player = this.player
        setPlayer(null)
        player?.release()
    }

    fun getPlayerState(): PlayerState? {
        return player?.let {
            PlayerState(
                it.currentPosition,
                it.volume,
                it.playWhenReady,
            )
        }
    }

    private fun setPlayer(player: ExoPlayer?) {
        if (this.player == player) {
            return
        }
        this.player?.removeListener(componentListener)
        this.player = player
        this.playerView?.player = player
        this.player?.addListener(componentListener)
        updateAll()
    }

    private fun createPlayer(): ExoPlayer {
        return ExoPlayer.Builder(context)
            .setAudioAttributes(AudioAttributes.DEFAULT, true)
            .build()
    }

    @androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
    private fun createMediaSource(uri: String): MediaSource {
        val cache = NovaVideoCacheManager.getSimpleCache(context)
        val factory = DefaultMediaSourceFactory(
            CacheDataSource.Factory()
                .setCache(cache)
                .setCacheWriteDataSinkFactory(CacheDataSink.Factory().setCache(cache))
                .setUpstreamDataSourceFactory(DefaultHttpDataSource.Factory())
                .setFlags(CacheDataSource.FLAG_IGNORE_CACHE_ON_ERROR)
        )
        return factory.createMediaSource(MediaItem.fromUri(uri))
    }

    private fun updateAll() {
        updatePlayPause()
        updateVolumeButton()
        updateTimeline()
    }


    private var currentPosition: Long = C.TIME_UNSET
    private var currentBufferedPosition: Long = C.TIME_UNSET
    private var duration: Long = C.TIME_UNSET
    private val updateProgressAction = this::updateProgress

    private fun updateTimeline() {
        val player = this.player ?: return

        val duration = player.contentDuration
        val durationChanged = duration != this.duration
        this.duration = duration
        if (durationChanged) {
            listener?.onDurationUpdate(duration)
        }

        updateProgress()
    }

    private fun updateProgress() {
        if (!isAttachedToWindow) {
            return
        }

        val player = this.player

        var position = player?.contentPosition ?: 0L
        var bufferedPosition = player?.contentBufferedPosition ?: 0L

        val positionChanged = position != currentPosition
        val bufferedPositionChanged = bufferedPosition != currentBufferedPosition
        currentPosition = position
        currentBufferedPosition = bufferedPosition

        if (positionChanged || bufferedPositionChanged) {
            listener?.onProgressUpdate(position, bufferedPosition)
        }

        // Cancel any pending updates and schedule a new one if necessary.
        removeCallbacks(updateProgressAction)
        val playbackState = player?.playbackState ?: Player.STATE_IDLE
        if (player != null && player.isPlaying) {
            var mediaTimeDelayMs = VideoPlayerController.MAX_UPDATE_INTERVAL_MS

            // Limit delay to the start of the next full second to ensure position display is smooth.
            val mediaTimeUntilNextFullSecondMs = 1000 - position % 1000
            mediaTimeDelayMs = mediaTimeDelayMs.coerceAtMost(mediaTimeUntilNextFullSecondMs)

            // Calculate the delay until the next update in real time, taking playback speed into account.
            val playbackSpeed = player.playbackParameters.speed
            var delayMs = if (playbackSpeed > 0) (mediaTimeDelayMs / playbackSpeed).toLong() else VideoPlayerController.MAX_UPDATE_INTERVAL_MS

            // Constrain the delay to avoid too frequent / infrequent updates.
            delayMs = delayMs.coerceIn(
                VideoPlayerController.DEFAULT_TIME_BAR_MIN_UPDATE_INTERVAL_MS,
                VideoPlayerController.MAX_UPDATE_INTERVAL_MS
            )
            postDelayed(updateProgressAction, delayMs)
        } else if (playbackState != Player.STATE_ENDED && playbackState != Player.STATE_IDLE) {
            postDelayed(updateProgressAction, VideoPlayerController.MAX_UPDATE_INTERVAL_MS)
        }
    }

    private fun updateVolumeButton() {
        if (!isAttachedToWindow) {
            return
        }

        val volume = player?.volume ?: 0F
        volumeOnButton?.visibility = if (volume > 0F) VISIBLE else GONE
        volumeOffButton?.visibility = if (volume > 0F) GONE else VISIBLE
    }

    private var shouldShowPauseButton = false

    private fun updatePlayPause() {
        if (!isAttachedToWindow) {
            return
        }

        val shouldShowPauseButton = shouldShowPauseButton()
        if (this.shouldShowPauseButton != shouldShowPauseButton) {
            this.shouldShowPauseButton = shouldShowPauseButton
            playButton?.visibility = if (shouldShowPauseButton) GONE else VISIBLE
            pauseButton?.visibility = if (shouldShowPauseButton) VISIBLE else GONE
        }
    }

    private fun shouldShowPauseButton(): Boolean {
        return player != null && player!!.playbackState != Player.STATE_ENDED && player!!.playbackState != Player.STATE_IDLE && player!!.playWhenReady
    }

    private inner class ComponentListener : Player.Listener, OnClickListener {

        private var isPlayingOrBuffering = false

        override fun onEvents(player: Player, events: Player.Events) {
            if (events.containsAny(
                    Player.EVENT_PLAYBACK_STATE_CHANGED,
                    Player.EVENT_PLAY_WHEN_READY_CHANGED
                )
            ) {
                updatePlayPause()
            }

            if (events.containsAny(
                    Player.EVENT_PLAYBACK_STATE_CHANGED,
                    Player.EVENT_PLAY_WHEN_READY_CHANGED,
                    Player.EVENT_IS_PLAYING_CHANGED
                )
            ) {
                updateProgress()
            }

            if (events.containsAny(
                    Player.EVENT_POSITION_DISCONTINUITY,
                    Player.EVENT_TIMELINE_CHANGED
                )
            ) {
                updateTimeline()
            }

            if (events.containsAny(
                    Player.EVENT_VOLUME_CHANGED,
                    Player.EVENT_RENDERED_FIRST_FRAME
                )
            ) {
                updateVolumeButton()
            }
        }

        override fun onPlayWhenReadyChanged(playWhenReady: Boolean, reason: Int) {
            updateIsPlayingOrBuffering()
        }

        override fun onPlaybackStateChanged(playbackState: Int) {
            updateIsPlayingOrBuffering()
            when (playbackState) {
                Player.STATE_ENDED -> {
                    listener?.onProgressUpdate(duration, duration)
                }
                else -> {}
            }
        }

        private fun updateIsPlayingOrBuffering() {
            val isPlayingOrBuffering = isPlayingOrBuffering()
            if (this.isPlayingOrBuffering != isPlayingOrBuffering) {
                this.isPlayingOrBuffering = isPlayingOrBuffering
                listener?.onIsPlayingOrBufferingChanged(isPlayingOrBuffering)
            }
        }

        private fun isPlayingOrBuffering(): Boolean {
            val player = player ?: return false
            return player.playbackState != Player.STATE_ENDED
                    && player.playbackState != Player.STATE_IDLE
                    && player.playWhenReady
        }

        override fun onPositionDiscontinuity(
            oldPosition: Player.PositionInfo,
            newPosition: Player.PositionInfo,
            reason: Int
        ) {
            if (reason == Player.DISCONTINUITY_REASON_AUTO_TRANSITION) {
                listener?.onProgressUpdate(duration, duration)
            }
        }

        override fun onClick(v: View?) {
            val player = player ?: return
            when (v) {
                playButton -> {
                    player.play()
                    listener?.onUserPlay()
                }
                pauseButton -> {
                    player.pause()
                    listener?.onUserPause()
                }
                volumeOnButton -> {
                    player.volume = 0F
                    listener?.onUserMute()
                }
                volumeOffButton -> {
                    player.volume = 1F
                    listener?.onUserUnmute()
                }
            }
        }
    }
}
