package com.particles.android.ads.internal.util.viewability

import android.view.View
import android.webkit.WebView
import com.iab.omid.library.newsbreak1.adsession.AdEvents
import com.iab.omid.library.newsbreak1.adsession.AdSession
import com.iab.omid.library.newsbreak1.adsession.FriendlyObstructionPurpose
import com.iab.omid.library.newsbreak1.adsession.media.Position
import com.iab.omid.library.newsbreak1.adsession.media.VastProperties
import com.particles.android.ads.internal.domain.AdVerification

class OMTracker(
    private val adView: View,
    private val adSession: AdSession,
    private val mediaExtras: MediaExtras? = null,
) {
    private val adEvents: AdEvents = AdEvents.createAdEvents(adSession)
    val mediaEvents: OMMediaEvents? = mediaExtras?.let { OMMediaEvents(adSession) }

    private var state = STATE_IDLE

    init {
        adSession.registerAdView(adView)
        OMSDKUtil.addFriendlyObstructions(adView, adSession)
    }

    fun addFriendlyObstruction(view: View, purpose: FriendlyObstructionPurpose, reason: String = "") {
        adSession.addFriendlyObstruction(view, purpose, reason)
    }

    fun removeFriendlyObstruction(view: View) {
        adSession.removeFriendlyObstruction(view)
    }

    fun startTracking() {
        changeState(STATE_STARTED)
    }

    fun stopTracking() {
        changeState(STATE_STOPPED)
    }

    fun trackImpression() {
        changeState(STATE_IMPRESSED)
    }

    private fun changeState(newState: Int) {
        when (newState) {
            STATE_STARTED -> if (state == STATE_IDLE) {
                adSession.start()
                if (mediaExtras != null) {
                    adEvents.loaded(createVastProperties(mediaExtras))
                } else {
                    adEvents.loaded()
                }
                state = newState
            }
            STATE_IMPRESSED -> if (state == STATE_STARTED) {
                adEvents.impressionOccurred()
                state = newState
            }
            STATE_STOPPED -> if (state != STATE_IDLE && state != STATE_STOPPED) {
                adSession.finish()
                state = newState
            }
        }
    }

    data class MediaExtras(
        val isAutoPlay: Boolean = false,
        val skipOffset: Float = -1F,
    )

    private fun createVastProperties(extras: MediaExtras): VastProperties {
        return if (extras.skipOffset >= 0)
            VastProperties.createVastPropertiesForSkippableMedia(extras.skipOffset, extras.isAutoPlay, Position.STANDALONE)
        else
            VastProperties.createVastPropertiesForNonSkippableMedia(extras.isAutoPlay, Position.STANDALONE)
    }

    companion object {
        const val STATE_IDLE = 0
        const val STATE_STARTED = 1
        const val STATE_IMPRESSED = 2
        const val STATE_STOPPED = 3

        fun createWebViewDisplayTracker(webView: WebView, adView: View = webView): OMTracker? {
            try {
                val adSession = OMSDKUtil.createHtmlAdSession(webView)
                if (adSession != null) {
                    return OMTracker(adView, adSession)
                }
            } catch (e: Exception) {
                // ignore
            }
            return null
        }

        fun createNativeDisplayTracker(adView: View, adVerifications: List<AdVerification>): OMTracker? {
            try {
                val urls = adVerifications.map { it.javascriptResourceUrl }
                val adSession = OMSDKUtil.createNativeAdSession(adView.context, urls)
                if (adSession != null) {
                    return OMTracker(adView, adSession)
                }
            } catch (e: Exception) {
                // ignore
            }
            return null
        }

        fun createNativeVideoTracker(adView: View, adVerifications: List<AdVerification>, extras: MediaExtras): OMTracker? {
            try {
                val urls = adVerifications.map { it.javascriptResourceUrl }
                val adSession = OMSDKUtil.createNativeVideoSession(adView.context, urls)
                if (adSession != null) {
                    return OMTracker(adView, adSession, extras)
                }
            } catch (e: Exception) {
                // ignore
            }
            return null
        }
    }
}
