package com.particles.android.ads.internal.util.viewability

import android.graphics.Rect
import android.view.View

internal class ViewabilityChecker(private val view: View) {
    private var firstVisibleNotified = false
    private var snapshot: Snapshot? = null
    private val counter = VisibleTimeCounter(100, 75, 50, 25, 1)

    private val tempRect = Rect()
    private val tempLocation = intArrayOf(0, 0)

    fun reset() {
        firstVisibleNotified = false
        snapshot = null
        counter.reset()
    }

    fun hasChanged(): Boolean {
        val snapshot = getViewabilitySnapshot()

        var changed = counter.update(snapshot.visiblePercent, snapshot.updatedAt)

        if (!firstVisibleNotified && snapshot.visiblePixels > 0) {
            firstVisibleNotified = true
            changed = true
        }

        this.snapshot = snapshot

        return changed
    }

    private fun getViewabilitySnapshot(): Snapshot {
        val displayMetrics = view.resources?.displayMetrics

        if (!view.isShown || view.windowVisibility != View.VISIBLE || !view.getGlobalVisibleRect(tempRect)) {
            tempRect.setEmpty()
        }

        view.getLocationOnScreen(tempLocation)

        return Snapshot(
            updatedAt = System.currentTimeMillis(),
            density = displayMetrics?.density ?: 0F,
            viewportWidth = displayMetrics?.widthPixels ?: 0,
            viewportHeight = displayMetrics?.heightPixels ?: 0,
            viewTop = tempLocation[1],
            viewLeft = tempLocation[0],
            viewRight = tempLocation[0] + view.width,
            viewBottom = tempLocation[1] + view.height,
            visibleTop = tempRect.top,
            visibleLeft = tempRect.left,
            visibleRight = tempRect.right,
            visibleBottom = tempRect.bottom,
        )
    }

    fun getState(): ViewabilityState {
        return ViewabilityState(
            viewportWidth = snapshot?.viewportWidth ?: 0,
            viewportHeight = snapshot?.viewportHeight ?: 0,
            viewTop = snapshot?.viewTop ?: 0,
            viewLeft = snapshot?.viewLeft ?: 0,
            viewRight = snapshot?.viewRight ?: 0,
            viewBottom = snapshot?.viewBottom ?: 0,
            visibleTop = snapshot?.visibleTop ?: 0,
            visibleLeft = snapshot?.visibleLeft ?: 0,
            visibleRight = snapshot?.visibleRight ?: 0,
            visibleBottom = snapshot?.visibleBottom ?: 0,
            visibleTime100 = counter.getVisibleTime(100),
            visibleTime75 = counter.getVisibleTime(75),
            visibleTime50 = counter.getVisibleTime(50),
            visibleTime25 = counter.getVisibleTime(25),
            visibleTime1 = counter.getVisibleTime(1),
        )
    }
}

private class Snapshot(
    val updatedAt: Long,
    val density: Float,
    val viewportWidth: Int,
    val viewportHeight: Int,
    val viewTop: Int,
    val viewLeft: Int,
    val viewRight: Int,
    val viewBottom: Int,
    val visibleTop: Int,
    val visibleLeft: Int,
    val visibleRight: Int,
    val visibleBottom: Int,
) {
    val visiblePixels: Long
        get() {
            val visibleWidth = visibleRight - visibleLeft
            val visibleHeight = visibleBottom - visibleTop
            return if (visibleWidth > 0 && visibleHeight > 0) {
                visibleWidth.toLong() * visibleHeight
            } else 0
        }

    val visiblePercent: Int
        get() {
            val viewWidth = viewRight - viewLeft
            val viewHeight = viewBottom - viewTop
            return if (viewWidth > 0 && viewHeight > 0) {
                (visiblePixels * 100 / viewWidth / viewHeight).toInt()
            } else 0
        }
}
