package com.particles.android.ads.nativead

import android.view.View
import com.particles.android.ads.internal.NativeAdImpl
import com.particles.android.ads.internal.rendering.addon.AddonViewHelper
import com.particles.android.ads.internal.util.AdViewHelper
import com.particles.android.ads.internal.util.viewability.OMTracker

internal class NativeAdViewHelper(private val view: NativeAdView) {
    private val adViewHelper = AdViewHelper(view)
    private var ad: NativeAdImpl? = null
    private val onClickListener = View.OnClickListener {
        if (it == view.mediaView && !isMediaViewClickable()) {
            return@OnClickListener
        }
        ad?.onAdClicked(it, getViewName(it))
    }

    private var omTracker: OMTracker? = null
    private fun createOMTracker(): OMTracker? {
        val adVerifications = ad?.creative?.adVerifications
        return if (!adVerifications.isNullOrEmpty()) {
            if (ad?.creativeType == NativeAd.CreativeType.VIDEO)
                OMTracker.createNativeVideoTracker(view, adVerifications, OMTracker.MediaExtras(
                    isAutoPlay = ad?.creative?.video?.isAutoPlay ?: false
                ))
            else
                OMTracker.createNativeDisplayTracker(view, adVerifications)
        } else null
    }

    fun register(ad: NativeAdImpl) {
        setOnClickListener(onClickListener)
        this.ad = ad
        omTracker = createOMTracker()
        omTracker?.startTracking()
        view.mediaView?.setNativeAd(ad, omTracker?.mediaEvents)
        view.addonView?.let {
            AddonViewHelper(it).setAddon(ad.adSession) {
                ad.addonListener?.onAddonDismiss()
            }
        }
        adViewHelper.register(ad)
        omTracker?.trackImpression()
    }

    fun unregister() {
        setOnClickListener(null)
        view.mediaView?.setNativeAd(null)
        adViewHelper.unregister()
        omTracker?.stopTracking()
        omTracker = null
        this.ad = null
    }

    private fun setOnClickListener(listener: View.OnClickListener?) {
        view.advertiserView?.setOnClickListener(listener)
        view.bodyView?.setOnClickListener(listener)
        view.callToActionView?.setOnClickListener(listener)
        view.headlineView?.setOnClickListener(listener)
        view.iconView?.setOnClickListener(listener)
        view.mediaView?.setOnClickListener(listener)
        view.starRatingView?.setOnClickListener(listener)
        view.addonView?.setOnClickListener(listener)
    }

    private fun getViewName(v: View): String? {
        return when (v) {
            view.advertiserView -> "advertiser"
            view.bodyView -> "body"
            view.callToActionView -> "cta"
            view.headlineView -> "headline"
            view.iconView -> "icon"
            view.mediaView -> "media"
            view.starRatingView -> "ratings"
            view.addonView -> "addon"
            else -> "unknown"
        }
    }

    private fun isMediaViewClickable(): Boolean {
        val ad = this.ad ?: return true

        return when (ad.creativeType) {
            NativeAd.CreativeType.IMAGE,
            NativeAd.CreativeType.SPONSORED_IMAGE -> ad.creative.isImageClickable

            NativeAd.CreativeType.VIDEO -> ad.creative.video?.isClickable == true

            else -> true
        }
    }
}
