package com.particles.prebidadapter

import android.content.Context
import android.content.res.Configuration
import android.text.TextUtils
import com.google.gson.Gson
import com.particles.msp.BidListener
import com.particles.msp.BidLoader
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.FacebookBidTokenListener
import com.particles.msp.adapter.FacebookBidTokenProvider
import com.particles.msp.adapter.GoogleQueryInfoFetcher
import com.particles.msp.adapter.GoogleQueryInfoListener
import com.particles.msp.api.AdFormat
import com.particles.msp.api.AdRequest
import com.particles.msp.api.MSPConstants
import com.particles.msp.util.Logger
import com.particles.msp.util.UserId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import org.prebid.mobile.AdSize
import org.prebid.mobile.BannerParameters
import org.prebid.mobile.NativeAdUnit
import org.prebid.mobile.NativeAsset
import org.prebid.mobile.NativeDataAsset
import org.prebid.mobile.NativeEventTracker
import org.prebid.mobile.NativeEventTracker.EVENT_TRACKING_METHOD
import org.prebid.mobile.NativeImageAsset
import org.prebid.mobile.NativeParameters
import org.prebid.mobile.NativeTitleAsset
import org.prebid.mobile.TargetingParams
import org.prebid.mobile.api.original.PrebidAdUnit
import org.prebid.mobile.api.original.PrebidRequest
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse

class PrebidBidLoader(val context: Context?,
                      googleQueryInfoFetcher: GoogleQueryInfoFetcher?,
                      facebookBidTokenProvider: FacebookBidTokenProvider?) :
    BidLoader(
        googleQueryInfoFetcher!!, facebookBidTokenProvider
    ) {
    fun createBannerParameters(adRequest: AdRequest): BannerParameters {
        var adSize = if (adRequest.adFormat == AdFormat.BANNER) AdSize(320, 50) else AdSize(300, 250)
        if (adRequest.adFormat == AdFormat.INTERSTITIAL) {
            val deviceConfiguration: Configuration = context!!.resources.configuration
            adSize = AdSize(deviceConfiguration.screenWidthDp, deviceConfiguration.screenHeightDp)
        }
        adRequest.adSize?.let {
            adSize = AdSize(it.width, it.height)
        }
        val parameters = BannerParameters()
        val adSizes: MutableSet<AdSize> = HashSet()
        adSizes.add(adSize)
        parameters.adSizes = adSizes
        return parameters
    }

    fun createNativeParameters(): NativeParameters {
        val assets: MutableList<NativeAsset> = ArrayList()
        val title = NativeTitleAsset()
        title.setLength(90) // max length of title, required.
        title.isRequired = true
        assets.add(title)
        val icon = NativeImageAsset(50, 50, 50, 50)
        icon.imageType = NativeImageAsset.IMAGE_TYPE.ICON
        icon.isRequired = true
        assets.add(icon)
        val image = NativeImageAsset(382, 200, 382, 200)
        image.imageType = NativeImageAsset.IMAGE_TYPE.MAIN
        image.isRequired = true
        assets.add(image)
        val data = NativeDataAsset()
        data.dataType = NativeDataAsset.DATA_TYPE.SPONSORED
        data.isRequired = true
        assets.add(data)
        val body = NativeDataAsset()
        body.isRequired = true
        body.dataType = NativeDataAsset.DATA_TYPE.DESC
        assets.add(body)
        val cta = NativeDataAsset()
        cta.isRequired = true
        cta.dataType = NativeDataAsset.DATA_TYPE.CTATEXT
        assets.add(cta)
        val nativeParameters = NativeParameters(assets)
        val methods = ArrayList<EVENT_TRACKING_METHOD>()
        methods.add(EVENT_TRACKING_METHOD.IMAGE)
        methods.add(EVENT_TRACKING_METHOD.JS)
        try {
            val tracker = NativeEventTracker(NativeEventTracker.EVENT_TYPE.IMPRESSION, methods)
            nativeParameters.addEventTracker(tracker)
        } catch (ignored: Exception) {
        }
        nativeParameters.setContextType(NativeAdUnit.CONTEXT_TYPE.CONTENT_CENTRIC)
        nativeParameters.setPlacementType(NativeAdUnit.PLACEMENTTYPE.CONTENT_FEED)
        return nativeParameters
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    override fun loadBid(
        placementId: String,
        adParams: Map<String, Any>,
        bidListener: BidListener,
        adRequest: AdRequest
    ) {
        googleQueryInfoFetcher.fetch(object : GoogleQueryInfoListener {
            override fun onComplete(queryInfo: String) {
                Logger.verbose("PrebidBidLoader. google queryInfo received: $queryInfo")
                CoroutineScope(Dispatchers.IO).launch {
                    val fbBidToken = facebookBidTokenProvider?.let { provider ->
                        suspendCancellableCoroutine<String?> { cont ->
                            provider.fetch(object : FacebookBidTokenListener {
                                override fun onComplete(bidToken: String) {
                                    cont.resume(bidToken, null)
                                }
                            }, adRequest.context)
                        }
                    }

                    // Reference: "buyeruid" in https://www.iab.com/wp-content/uploads/2016/03/OpenRTB-API-Specification-Version-2-5-FINAL.pdf
                    // for Prebid server & FB Audience network server-to-server integration
                    fbBidToken?.let { TargetingParams.setBuyerId(it) }

                    val prebidAdUnit = PrebidAdUnit(placementId)
                    val prebidRequest = PrebidRequest()
                    prebidRequest.setBannerParameters(createBannerParameters(adRequest))

                    if (adRequest.adFormat == AdFormat.MULTI_FORMAT || adRequest.adFormat == AdFormat.NATIVE) {
                        prebidRequest.setNativeParameters(createNativeParameters())
                    }

                    if (adRequest.adFormat == AdFormat.INTERSTITIAL) {
                        prebidRequest.setInterstitial(true)
                    }

                    val customParams: MutableMap<String, Any> = adRequest.customParams.toMutableMap()
                    if (!TextUtils.isEmpty(queryInfo)) {
                        customParams["query_info"] = queryInfo
                    }

                    if (MSPConstants.CUSTOM_PARAM_KEY_USER_ID !in customParams) {
                        UserId.getCachedUserId()?.let {
                            customParams[MSPConstants.CUSTOM_PARAM_KEY_USER_ID] = it
                        }
                    }

                    val gson = Gson()

                    // only add test params in debug build
                    fun isDebugBuild(): Boolean {
                        return try {
                            val appPackageName = context!!.applicationContext.packageName
                            val appBuildConfig = Class.forName("$appPackageName.BuildConfig")
                            val debugField = appBuildConfig.getField("DEBUG")
                            debugField.getBoolean(null) // Get the static value of BuildConfig.DEBUG
                        } catch (e: Exception) {
                            false
                        }
                    }
                    if (isDebugBuild()) {
                        customParams["test"] = gson.toJson(adRequest.testParams)
                    }

                    customParams["app_install_timestamp"] = getAppInstallTime(adRequest.context as? Context)
                    addCustomTargeting(prebidRequest, customParams)
                    Logger.debug("PrebidBidLoader. send bid request to MSP server")

                    prebidAdUnit.fetchDemand(prebidRequest) { bidInfo, message ->
                        Logger.debug("PrebidBidLoader. Bid response received. price: ${bidInfo.bidResponse?.winningBid?.price}")
                        val bidResponse = bidInfo.bidResponse
                        bidResponse?.let {
                            bidListener.onBidResponse(it, getAdNetwork(it))
                        } ?: run {
                            bidListener.onError(bidInfo.resultCode.toString() + ". " + message)
                        }
                    }
                }
            }
        }, adRequest)
    }



    fun getAdNetwork(bidResponse: BidResponse): AdNetwork {
        return when(bidResponse.winningBid?.prebid?.targeting?.get("hb_bidder")) {
            "msp_google" -> AdNetwork.Google
            "audienceNetwork" -> AdNetwork.Facebook
            "msp_nova" -> AdNetwork.Nova
            else -> AdNetwork.Prebid
        }
    }

    companion object {
        fun addCustomTargeting(prebidRequest: PrebidRequest, params: Map<String, Any>) {
            val parametersToSkip = setOf(
                MSPConstants.AD_REQUEST_CUSTOM_PARAM_KEY_GOOGLE_MULTI_FORMAT,
                MSPConstants.GOOGLE_AD_MULTI_CONTENT_URLS,
                MSPConstants.GOOGLE_AD_CONTENT_URL
            )
            val contextData: MutableMap<String, Set<String>> = HashMap()
            for ((key, value) in params) {
                if (key in parametersToSkip) {
                    continue
                }

                if (value is List<*>) {
                    if (value.isNotEmpty()) {
                        contextData[key] = HashSet(value.map { it.toString() })
                    }
                } else value?.let{
                    contextData[key] = HashSet(setOf(it.toString()))
                }
            }
            prebidRequest.setExtData(contextData)
        }
    }
}
