package com.particles.prebidadapter

import android.Manifest
import android.content.Context
import android.content.res.Configuration
import android.text.TextUtils
import androidx.annotation.RequiresPermission
import com.google.gson.Gson
import com.particles.msp.BidListener
import com.particles.msp.BidLoader
import com.particles.msp.MSPManager
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.BidTokenProvider
import com.particles.msp.adapter.fetchAsync
import com.particles.msp.api.AdFormat
import com.particles.msp.api.AdRequest
import com.particles.msp.api.MSPConstants
import com.particles.msp.util.Logger
import com.particles.msp.util.UserId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withTimeoutOrNull
import org.prebid.mobile.AdSize
import org.prebid.mobile.BannerParameters
import org.prebid.mobile.NativeAdUnit
import org.prebid.mobile.NativeAsset
import org.prebid.mobile.NativeDataAsset
import org.prebid.mobile.NativeEventTracker
import org.prebid.mobile.NativeEventTracker.EVENT_TRACKING_METHOD
import org.prebid.mobile.NativeImageAsset
import org.prebid.mobile.NativeParameters
import org.prebid.mobile.NativeTitleAsset
import org.prebid.mobile.TargetingParams
import org.prebid.mobile.api.original.PrebidAdUnit
import org.prebid.mobile.api.original.PrebidRequest
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse

class PrebidBidLoader(
    val context: Context?,
    googleQueryInfoFetcher: BidTokenProvider?,
    facebookBidTokenProvider: BidTokenProvider?,
    molocoBidTokenProvider: BidTokenProvider?,
) : BidLoader(googleQueryInfoFetcher, facebookBidTokenProvider, molocoBidTokenProvider) {
    fun createBannerParameters(adRequest: AdRequest): BannerParameters {
        var adSize = if (adRequest.adFormat == AdFormat.BANNER) AdSize(320, 50) else AdSize(300, 250)
        if (adRequest.adFormat == AdFormat.INTERSTITIAL) {
            val deviceConfiguration: Configuration = context!!.resources.configuration
            adSize = AdSize(deviceConfiguration.screenWidthDp, deviceConfiguration.screenHeightDp)
        }
        adRequest.adSize?.let {
            adSize = AdSize(it.width, it.height)
        }
        val parameters = BannerParameters()
        val adSizes: MutableSet<AdSize> = HashSet()
        adSizes.add(adSize)
        parameters.adSizes = adSizes
        return parameters
    }

    fun createNativeParameters(): NativeParameters {
        val assets: MutableList<NativeAsset> = ArrayList()
        val title = NativeTitleAsset()
        title.setLength(90) // max length of title, required.
        title.isRequired = true
        assets.add(title)
        val icon = NativeImageAsset(50, 50, 50, 50)
        icon.imageType = NativeImageAsset.IMAGE_TYPE.ICON
        icon.isRequired = true
        assets.add(icon)
        val image = NativeImageAsset(382, 200, 382, 200)
        image.imageType = NativeImageAsset.IMAGE_TYPE.MAIN
        image.isRequired = true
        assets.add(image)
        val data = NativeDataAsset()
        data.dataType = NativeDataAsset.DATA_TYPE.SPONSORED
        data.isRequired = true
        assets.add(data)
        val body = NativeDataAsset()
        body.isRequired = true
        body.dataType = NativeDataAsset.DATA_TYPE.DESC
        assets.add(body)
        val cta = NativeDataAsset()
        cta.isRequired = true
        cta.dataType = NativeDataAsset.DATA_TYPE.CTATEXT
        assets.add(cta)
        val nativeParameters = NativeParameters(assets)
        val methods = ArrayList<EVENT_TRACKING_METHOD>()
        methods.add(EVENT_TRACKING_METHOD.IMAGE)
        methods.add(EVENT_TRACKING_METHOD.JS)
        try {
            val tracker = NativeEventTracker(NativeEventTracker.EVENT_TYPE.IMPRESSION, methods)
            nativeParameters.addEventTracker(tracker)
        } catch (ignored: Exception) {
        }
        nativeParameters.setContextType(NativeAdUnit.CONTEXT_TYPE.CONTENT_CENTRIC)
        nativeParameters.setPlacementType(NativeAdUnit.PLACEMENTTYPE.CONTENT_FEED)
        return nativeParameters
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    override fun loadBid(
        placementId: String,
        adParams: Map<String, Any>,
        bidListener: BidListener,
        adRequest: AdRequest
    ) {
        fun makeBidRequest(fbBidToken: String?, queryInfo: String?, molocoBidToken: String?) {
            // Reference: "buyeruid" in https://www.iab.com/wp-content/uploads/2016/03/OpenRTB-API-Specification-Version-2-5-FINAL.pdf
            // for Prebid server & FB Audience network server-to-server integration
            TargetingParams.setBuyerId(fbBidToken);

            val prebidAdUnit = PrebidAdUnit(placementId)
            val prebidRequest = PrebidRequest()
            prebidRequest.setBannerParameters(createBannerParameters(adRequest))

            if (adRequest.adFormat == AdFormat.MULTI_FORMAT || adRequest.adFormat == AdFormat.NATIVE) {
                prebidRequest.setNativeParameters(createNativeParameters())
            }

            if (adRequest.adFormat == AdFormat.INTERSTITIAL) {
                prebidRequest.setInterstitial(true)
            }

            val customParams: MutableMap<String, Any> = adRequest.customParams.toMutableMap()
            if (!TextUtils.isEmpty(queryInfo)) {
                customParams["query_info"] = queryInfo!!
            }
            if (!molocoBidToken.isNullOrEmpty()) {
                customParams["moloco_bid_token"] = molocoBidToken
            }

            if (MSPConstants.CUSTOM_PARAM_KEY_USER_ID !in customParams) {
                UserId.getCachedUserId()?.let {
                    customParams[MSPConstants.CUSTOM_PARAM_KEY_USER_ID] = it
                }
            }

            if (MSPConstants.USER_SIGNAL_APP_INSTALL_TIME !in customParams) {
                customParams[MSPConstants.USER_SIGNAL_APP_INSTALL_TIME] = getAppInstallTime(adRequest.context)
            }

            val gson = Gson()

            customParams["test"] = gson.toJson(adRequest.testParams)

            MSPManager.adapterMetadataMap
                ?.map { (key, value) -> key.name.lowercase() to value.sdkVersion }
                ?.toMap()
                ?.let { customParams["adn_sdk_versions"] = gson.toJson(it) }

            customParams[DEVICE_SIGNAL_AVAILABLE_MEMORY] = getAvailableMemory(adRequest.context)
            customParams[DEVICE_SIGNAL_FONT_SIZE] = getFontSize(adRequest.context)
            customParams[DEVICE_SIGNAL_IS_LOW_DATA_MODE] = getIsLowDataMode(adRequest.context)
            customParams[DEVICE_SIGNAL_IS_LOW_POWER_MODE] = getIsLowPowerMode(adRequest.context)
            customParams[DEVICE_SIGNAL_ORIENTATION] = getOrientation(adRequest.context)
            customParams[DEVICE_SIGNAL_IS_IN_FOREGROUND] = getIsInForeground()
            getBatteryStatus(adRequest.context, customParams)

            addCustomTargeting(prebidRequest, customParams)
            Logger.verbose("PrebidBidLoader. send bid request $prebidRequest with custom params: $customParams")

            prebidAdUnit.fetchDemand(prebidRequest) { bidInfo, message ->
                Logger.verbose("PrebidBidLoader. Bid response received. price: ${bidInfo.bidResponse?.winningBid?.price}")
                val bidResponse = bidInfo.bidResponse
                bidResponse?.let {
                    bidListener.onBidResponse(it, getAdNetwork(it))
                } ?: run {
                    bidListener.onError(bidInfo.resultCode.toString() + ". " + message)
                }
            }
        }

        if (adRequest.customParams[MSPConstants.REQUEST_PARAM_KEY_NOVA_ONLY] as? Boolean == true) {
            makeBidRequest(null, null, null)
        } else {
            CoroutineScope(Dispatchers.IO).launch {
                val (ggQueryInfo, fbBidToken, molocoBidToken) = fetchBidTokens(adRequest, 10000)
                makeBidRequest(fbBidToken, ggQueryInfo, molocoBidToken)
            }
        }
    }

    private suspend fun fetchBidTokens(adRequest: AdRequest, timeoutMillis: Long): Triple<String?, String?, String?> = coroutineScope {
        val d1 = async(Dispatchers.IO) { withTimeoutOrNull(timeoutMillis) { fetchGoogleBidToken(adRequest) } }
        val d2 = async(Dispatchers.IO) { withTimeoutOrNull(timeoutMillis) { fetchFacebookBidToken(adRequest)} }
        val d3 = async(Dispatchers.IO) { withTimeoutOrNull(timeoutMillis) { fetchMolocoBidToken(adRequest) } }

        Triple(d1.await(), d2.await(), d3.await())
    }

    private suspend fun fetchGoogleBidToken(adRequest: AdRequest): String {
        return googleQueryInfoFetcher?.fetchAsync(adRequest) ?: ""
    }

    private suspend fun fetchFacebookBidToken(adRequest: AdRequest): String {
        return facebookBidTokenProvider?.fetchAsync(adRequest) ?: ""
    }

    private suspend fun fetchMolocoBidToken(adRequest: AdRequest): String {
        return molocoBidTokenProvider?.fetchAsync(adRequest) ?: ""
    }

    fun getAdNetwork(bidResponse: BidResponse): AdNetwork {
        return when(bidResponse.winningBid?.prebid?.targeting?.get("hb_bidder")) {
            "msp_google" -> AdNetwork.Google
            "audienceNetwork" -> AdNetwork.Facebook
            "msp_nova" -> AdNetwork.Nova
            "msp_moloco" -> AdNetwork.Moloco
            else -> AdNetwork.Prebid
        }
    }

    companion object {
        fun addCustomTargeting(prebidRequest: PrebidRequest, params: Map<String, Any>) {
            val parametersToSkip = setOf(
                MSPConstants.AD_REQUEST_CUSTOM_PARAM_KEY_GOOGLE_MULTI_FORMAT,
                MSPConstants.GOOGLE_AD_MULTI_CONTENT_URLS,
                MSPConstants.GOOGLE_AD_CONTENT_URL
            )
            val contextData: MutableMap<String, Set<String>> = HashMap()
            for ((key, value) in params) {
                if (key in parametersToSkip) {
                    continue
                }

                if (value is List<*>) {
                    if (value.isNotEmpty()) {
                        contextData[key] = HashSet(value.map { it.toString() })
                    }
                } else value?.let{
                    contextData[key] = HashSet(setOf(it.toString()))
                }
            }
            prebidRequest.setExtData(contextData)
        }
    }
}
