package com.particles.prebidadapter

import android.content.Context
import com.particles.msp.AdCache
import com.particles.msp.BidListener
import com.particles.msp.BidLoader
import com.particles.msp.MSPManager
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.api.MSPAd
import com.particles.msp.auction.AuctionBid
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.auction.BidderInfo
import com.particles.msp.util.Logger
import com.particles.msp.util.MesTrackerExt.trackAdRequest
import com.particles.msp.util.MesTrackerExt.trackAdResponse
import com.particles.msp.util.getBuckets
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse

class PrebidAdLoader {
    private var bidLoader: BidLoader? = null
    var adNetworkAdapter: AdNetworkAdapter? = null

    fun loadAd(bidderPlacementId: String, auctionBidListener: AuctionBidListener, context: Context, adRequest: AdRequest, adListener: AdListener, bidderInfo: BidderInfo) {
        val adResponseStartTime = System.currentTimeMillis()
        class BidListenerImp : BidListener {
            override fun onBidResponse(bidResponse: BidResponse, adNetwork: AdNetwork) {
                Logger.info("[Bidder: Prebid] Loading creative from adNetwork: $adNetwork ...")
                adNetworkAdapter = MSPManager.adNetworkAdapterProvider?.getAdNetworkAdapter(adNetwork)

                val extendedAuctionBidListener = object: AuctionBidListener {
                    override fun onSuccess(bid: AuctionBid) {
                        trackAdResponse(bid.bidderPlacementId, adRequest, bidResponse, adResponseStartTime)
                        auctionBidListener.onSuccess(bid)
                    }

                    override fun onError(error: String) {
                        trackAdResponseFailed(adRequest, bidResponse, adResponseStartTime, error)
                        auctionBidListener.onError(error)
                    }
                }

                adNetworkAdapter?.loadAdCreative(bidResponse, extendedAuctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo) ?: run {
                    val errorMessage = "[Bidder: Prebid] adNetwork Adapter is null for network: $adNetwork"
                    trackAdResponseFailed(adRequest, bidResponse, adResponseStartTime, errorMessage)
                    auctionBidListener.onError(errorMessage)
                    Logger.info("[Bidder: Prebid] adNetwork Adapter is null for network: $adNetwork")
                }
            }

            override fun onError(msg: String) {
                auctionBidListener.onError("[Bidder: Prebid] Prebid bid request error: $msg")
                Logger.info("[Bidder: Prebid] Prebid bid request error: $msg")
                trackAdResponseFailed(adRequest, null, adResponseStartTime, msg)
            }
        }
        bidLoader = MSPManager.bidLoaderProvider?.getBidLoader()
        Logger.verbose("[Bidder: Prebid] sending bid request for bidderPlacementId: $bidderPlacementId")
        bidLoader?.loadBid(bidderPlacementId, mapOf(), BidListenerImp(), adRequest)
        Logger.verbose("[Bidder: Prebid] track ad request event")
        MSPManager.mesTracker?.trackAdRequest(adRequest)
    }

    private fun trackAdResponse(
        bidderPlacementId: String,
        adRequest: AdRequest,
        bidResponse: BidResponse,
        adResponseStartTime: Long
    ) {
        val ad = AdCache.peakAd(bidderPlacementId)
        if (ad == null) {
            trackAdResponseFailed(
                adRequest,
                bidResponse,
                adResponseStartTime,
                "Failed to get ad from cache"
            )
        } else {
            trackAdResponseSuccess(
                adRequest,
                bidResponse,
                ad,
                adResponseStartTime
            )
        }
    }

    private fun trackAdResponseSuccess(
        adRequest: AdRequest,
        bidResponse: BidResponse,
        ad: MSPAd,
        adResponseStartTime: Long,
    ) {
        Logger.verbose("[Bidder: Prebid] track ad response success event")
        MSPManager.mesTracker?.trackAdResponse(
            adRequest,
            bidResponse.winningBid,
            ad,
            getBuckets(bidResponse.ext.map),
            (System.currentTimeMillis() - adResponseStartTime).toInt(),
            com.particles.mes.protos.ErrorCode.ERROR_CODE_SUCCESS
        )
    }

    private fun trackAdResponseFailed(
        adRequest: AdRequest,
        bidResponse: BidResponse?,
        adResponseStartTime: Long,
        errorMessage: String? = null,
    ) {
        Logger.verbose("[Bidder: Prebid] track ad response failed event. errorMessage = $errorMessage")
        MSPManager.mesTracker?.trackAdResponse(
            adRequest,
            bidResponse?.winningBid,
            null,
            getBuckets(bidResponse?.ext?.map),
            (System.currentTimeMillis() - adResponseStartTime).toInt(),
            com.particles.mes.protos.ErrorCode.ERROR_CODE_NO_FILL,
            errorMessage
        )
    }
}