package com.particles.prebidadapter

import android.content.Context
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.AdFormat
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.api.BannerAdView
import com.particles.msp.api.NativeAd
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.auction.BidderInfo
import com.particles.msp.util.Logger
import com.particles.msp.util.getBidder
import org.prebid.mobile.AdSize
import org.prebid.mobile.Host
import org.prebid.mobile.PrebidMobile
import org.prebid.mobile.api.data.AdUnitFormat
import org.prebid.mobile.api.exceptions.AdException
import org.prebid.mobile.api.exceptions.InitError
import org.prebid.mobile.api.rendering.BannerView
import org.prebid.mobile.api.rendering.listeners.BannerViewListener
import org.prebid.mobile.configuration.AdUnitConfiguration
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import org.prebid.mobile.rendering.bidding.display.InterstitialController
import org.prebid.mobile.rendering.bidding.interfaces.InterstitialControllerListener
import org.prebid.mobile.rendering.listeners.SdkInitializationListener
import org.prebid.mobile.rendering.models.AdPosition
import java.util.EnumSet

class PrebidAdapter : AdNetworkAdapter() {
    private var bannerView: BannerView? = null
    private var interstitialController: InterstitialController? = null
    private val tagPrefix = "[Adapter: Prebid]"
    override val adNetworkName: String
        get() {
            return bidResponse?.winningBid?.jsonString?.let {
                getBidder(it)
            } ?: "unknown"
        }
    override val adNetwork: AdNetwork = AdNetwork.Prebid

    override fun loadAdCreative(
        bidResponse: BidResponse?,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: AdRequest,
        adListener: AdListener,
        bidderPlacementId: String,
        bidderInfo: BidderInfo,
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        val adUnitId = adRequest.placementId

        val winnerBid = bidResponse?.winningBid
        if (winnerBid == null) {
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load prebid Ad: No Winning bid returned")
            return
        }

        when (adRequest.adFormat) {
            AdFormat.BANNER -> loadBannerAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.NATIVE -> loadNativeAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.INTERSTITIAL -> loadInterstitialAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.MULTI_FORMAT -> when (val type = winnerBid.prebid.type) {
                "banner" -> loadBannerAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
                "native" -> loadNativeAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
                else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load prebid Ad: Unsupported Ad type: $type")
            }
            else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load prebid Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: AdRequest,
        adListener: AdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String,
    ) {
        var adSize = AdSize(320, 50)
        adRequest.adSize?.let {
            adSize = AdSize(it.width, it.height)
        }
        val bannerView = BannerView(context, adRequest.placementId, adSize)
        this.bannerView = bannerView
        bannerView.bidResponse = bidResponse
        bannerView.setBannerListener(object : BannerViewListener {
            override fun onBidResponse(bannerView: BannerView?) {}

            override fun onAdLoaded(bannerView: BannerView?) {
                val bannerAdView = BannerAdView(bannerView!!, this@PrebidAdapter)
                Logger.info("$tagPrefix Prebid banner ad loaded successfully.")
                handleAdLoaded(bannerAdView, auctionBidListener, bidderPlacementId, bidderPlacementId, bidResponse?.winningBid?.price?: 0.0)
            }

            override fun onAdDisplayed(bannerView: BannerView?) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdFailed(bannerView: BannerView?, exception: AdException?) {}

            override fun onAdClicked(bannerView: BannerView?) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdClosed(bannerView: BannerView?) {}
        })
        bannerView.requestAd()
    }

    private fun loadNativeAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: AdRequest,
        adListener: AdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String,
    ) {
        Logger.info("$tagPrefix Failed to load Prebid native Ad: not supported")
        handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Prebid native Ad: not supported")
    }

    private fun loadInterstitialAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: AdRequest,
        adListener: AdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String,
    ) {
        Logger.info("$tagPrefix start to load prebid interstitial ads: $bidderPlacementId")

        val adUnitConfig = AdUnitConfiguration().apply {
            configId = adUnitId
            setAdUnitFormats(EnumSet.of(AdUnitFormat.BANNER, AdUnitFormat.VIDEO))
            setAdPosition(AdPosition.FULLSCREEN)
        }

        try {
            interstitialController = InterstitialController(context, object: InterstitialControllerListener {
                override fun onInterstitialReadyForDisplay() {
                    Logger.info("$tagPrefix successfully loaded Prebid interstitial ad")
                    val interstitialController = interstitialController ?: return
                    val mspInterstitialAd = PrebidInterstitialAd(this@PrebidAdapter, interstitialController)
                    handleAdLoaded(mspInterstitialAd, auctionBidListener, bidderPlacementId, adUnitId, winnerBid.price)
                }

                override fun onInterstitialClicked() {
                    handleAdClicked(adListener, adRequest)
                }

                override fun onInterstitialFailedToLoad(e: AdException?) {
                    Logger.info("$tagPrefix Failed to load Prebid interstitial Ad: ${e?.message}")
                    handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Prebid interstitial Ad: ${e?.message}")
                }

                override fun onInterstitialDisplayed() {
                    handleAdImpression(adListener, adRequest)
                }

                override fun onInterstitialClosed() {
                    handleAdDismissed(adListener, adRequest)
                }

            })
            interstitialController?.loadAd(adUnitConfig, bidResponse)
        } catch (e: AdException) {
            Logger.info("$tagPrefix Failed to load Prebid interstitial Ad: ${e.message}")
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Prebid interstitial Ad: ${e.message}")
        }
    }

    override fun prepareViewForInteraction(nativeAd: NativeAd, nativeAdView: Any) {
    }

    override fun destroyAd() {
        bannerView?.setBannerListener(null)
        bannerView?.destroy()
        interstitialController?.destroy()
        interstitialController = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context,
    ) {
        super.initialize(initParams, adapterInitListener, context)
        // Prebid SDk must be initialized in UI main thread.
        PrebidMobile.setPrebidServerAccountId(initParams.getPrebidAPIKey())
        Host.CUSTOM.hostUrl = initParams.getPrebidHostUrl()
        PrebidMobile.setPrebidServerHost(Host.CUSTOM)
        PrebidMobile.setShareGeoLocation(true)
        PrebidMobile.setTimeoutMillis(10000)
        PrebidMobile.initializeSdk(context, object : SdkInitializationListener {
            override fun onSdkInit() {
                adapterInitListener.onComplete(AdNetwork.Prebid, AdapterInitStatus.SUCCESS, AdapterInitStatus.SUCCESS.message)
            }

            override fun onSdkFailedToInit(error: InitError?) {}
        })
    }
}
