/*
 *    Copyright 2018-2019 Prebid.org, Inc.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.prebid.mobile;

import android.content.Context;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.text.TextUtils;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import org.jetbrains.annotations.NotNull;
import org.prebid.mobile.api.data.AdFormat;
import org.prebid.mobile.api.data.BidInfo;
import org.prebid.mobile.api.data.FetchDemandResult;
import org.prebid.mobile.api.exceptions.AdException;
import org.prebid.mobile.api.original.OnFetchDemandResult;
import org.prebid.mobile.configuration.AdUnitConfiguration;
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse;
import org.prebid.mobile.rendering.bidding.listeners.BidRequesterListener;
import org.prebid.mobile.rendering.bidding.loader.BidLoader;
import org.prebid.mobile.tasksmanager.TasksManager;

import java.util.*;

import static org.prebid.mobile.PrebidMobile.AUTO_REFRESH_DELAY_MAX;
import static org.prebid.mobile.PrebidMobile.AUTO_REFRESH_DELAY_MIN;

public abstract class AdUnit {

    protected AdUnitConfiguration configuration = new AdUnitConfiguration();

    @Nullable
    protected BidLoader bidLoader;
    @Nullable
    protected Object adObject;

    @Nullable
    protected BidResponse bidResponse;

    protected boolean allowNullableAdObject = false;

    AdUnit(@NonNull String configId, @NonNull EnumSet<AdFormat> adTypes) {
        configuration.setConfigId(configId);
        configuration.setAdFormats(adTypes);
        configuration.setIsOriginalAdUnit(true);
    }

    public AdUnit(@NotNull String configId) {
        configuration.setConfigId(configId);
        configuration.setIsOriginalAdUnit(true);
    }

    /**
     * @deprecated Please use setAutoRefreshInterval() in seconds!
     */
    @Deprecated
    public void setAutoRefreshPeriodMillis(
            @IntRange(from = AUTO_REFRESH_DELAY_MIN, to = AUTO_REFRESH_DELAY_MAX) int periodMillis
    ) {
        configuration.setAutoRefreshDelay(periodMillis / 1000);
    }

    public void setAutoRefreshInterval(
            @IntRange(from = AUTO_REFRESH_DELAY_MIN / 1000, to = AUTO_REFRESH_DELAY_MAX / 1000) int seconds
    ) {
        configuration.setAutoRefreshDelay(seconds);
    }

    public void resumeAutoRefresh() {
        LogUtil.verbose("Resuming auto refresh...");
        if (bidLoader != null) {
            bidLoader.setupRefreshTimer();
        }
    }

    public void stopAutoRefresh() {
        LogUtil.verbose("Stopping auto refresh...");
        if (bidLoader != null) {
            bidLoader.cancelRefresh();
        }
    }

    public void destroy() {
        if (bidLoader != null) {
            bidLoader.destroy();
        }
    }

    public void fetchDemand(@NonNull final OnCompleteListener2 listener) {
        final Map<String, String> keywordsMap = new HashMap<>();

        fetchDemand(keywordsMap, (resultCode, message) -> {
            TasksManager.getInstance().executeOnMainThread(() ->
                    listener.onComplete(resultCode, keywordsMap.size() != 0 ? Collections.unmodifiableMap(keywordsMap) : null, message)
            );
        });
    }

    public void fetchDemand(Object adObj, @NonNull OnCompleteListener listener) {
        if (TextUtils.isEmpty(PrebidMobile.getPrebidServerAccountId())) {
            LogUtil.error("Empty account id.");
            listener.onComplete(ResultCode.INVALID_ACCOUNT_ID, null);
            return;
        }
        if (TextUtils.isEmpty(configuration.getConfigId())) {
            LogUtil.error("Empty config id.");
            listener.onComplete(ResultCode.INVALID_CONFIG_ID, null);
            return;
        }
        if (PrebidMobile.getPrebidServerHost().equals(Host.CUSTOM)) {
            if (TextUtils.isEmpty(PrebidMobile.getPrebidServerHost().getHostUrl())) {
                LogUtil.error("Empty host url for custom Prebid Server host.");
                listener.onComplete(ResultCode.INVALID_HOST_URL, null);
                return;
            }
        }

        HashSet<AdSize> sizes = configuration.getSizes();
        for (AdSize size : sizes) {
            if (size.getWidth() < 0 || size.getHeight() < 0) {
                listener.onComplete(ResultCode.INVALID_SIZE, null);
                return;
            }
        }

        Context context = PrebidMobile.getApplicationContext();
        if (context != null) {
            ConnectivityManager conMgr = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (conMgr != null && context.checkCallingOrSelfPermission("android.permission.ACCESS_NETWORK_STATE") == PackageManager.PERMISSION_GRANTED) {
                NetworkInfo activeNetworkInfo = conMgr.getActiveNetworkInfo();
                if (activeNetworkInfo == null || !activeNetworkInfo.isConnected()) {
                    listener.onComplete(ResultCode.NETWORK_ERROR, null);
                    return;
                }
            }
        } else {
            LogUtil.error("Invalid context");
            listener.onComplete(ResultCode.INVALID_CONTEXT, null);
            return;
        }

        if (Util.supportedAdObject(adObj) || allowNullableAdObject) {
            adObject = adObj;
            bidLoader = new BidLoader(
                    context,
                    configuration,
                    createBidListener(listener)
            );

            if (configuration.getAutoRefreshDelay() > 0) {
                BidLoader.BidRefreshListener bidRefreshListener = () -> true;
                bidLoader.setBidRefreshListener(bidRefreshListener);
                LogUtil.verbose("Start fetching bids with auto refresh millis: " + configuration.getAutoRefreshDelay());
            } else {
                bidLoader.setBidRefreshListener(null);
                LogUtil.verbose("Start a single fetching.");
            }

            bidLoader.load();
        } else {
            adObject = null;
            listener.onComplete(ResultCode.INVALID_AD_OBJECT, null);
        }

    }


    public void fetchDemand(OnFetchDemandResult listener) {
        if (listener == null) {
            LogUtil.error("Parameter OnFetchDemandResult in fetchDemand() must be not null.");
            return;
        }

        allowNullableAdObject = true;

        fetchDemand(null, (resultCode, message) -> {
                BidInfo bidInfo = BidInfo.create(resultCode, bidResponse, configuration);
                Util.saveCacheId(bidInfo.getNativeCacheId(), adObject);
                listener.onComplete(bidInfo, message);
        });
    }

    // MARK: - adunit context data aka inventory data (imp[].ext.context.data)

    /**
     * This method obtains the context data keyword & value for adunit context targeting
     * if the key already exists the value will be appended to the list. No duplicates will be added
     */
    public void addContextData(String key, String value) {
        configuration.addContextData(key, value);
    }

    /**
     * This method obtains the context data keyword & values for adunit context targeting
     * the values if the key already exist will be replaced with the new set of values
     */
    public void updateContextData(String key, Set<String> value) {
        configuration.addContextData(key, value);
    }

    /**
     * This method allows to remove specific context data keyword & values set from adunit context targeting
     */
    public void removeContextData(String key) {
        configuration.removeContextData(key);
    }

    /**
     * This method allows to remove all context data set from adunit context targeting
     */
    public void clearContextData() {
        configuration.clearContextData();
    }

    Map<String, Set<String>> getContextDataDictionary() {
        return configuration.getContextDataDictionary();
    }

    // MARK: - adunit context keywords (imp[].ext.context.keywords)

    /**
     * This method obtains the context keyword for adunit context targeting
     * Inserts the given element in the set if it is not already present.
     */
    public void addContextKeyword(String keyword) {
        configuration.addContextKeyword(keyword);
    }

    /**
     * This method obtains the context keyword set for adunit context targeting
     * Adds the elements of the given set to the set.
     */
    public void addContextKeywords(Set<String> keywords) {
        configuration.addContextKeywords(keywords);
    }

    /**
     * This method allows to remove specific context keyword from adunit context targeting
     */
    public void removeContextKeyword(String keyword) {
        configuration.removeContextKeyword(keyword);
    }

    /**
     * This method allows to remove all keywords from the set of adunit context targeting
     */
    public void clearContextKeywords() {
        configuration.clearContextKeywords();
    }

    Set<String> getContextKeywordsSet() {
        return configuration.getContextKeywordsSet();
    }

    /**
     * This method obtains the content for adunit, content, in which impression will appear
     */
    public void setAppContent(ContentObject content) {
        configuration.setAppContent(content);
    }

    public ContentObject getAppContent() {
        return configuration.getAppContent();
    }

    public void addUserData(DataObject dataObject) {
        configuration.addUserData(dataObject);
    }

    public ArrayList<DataObject> getUserData() {
        return configuration.getUserData();
    }

    public void clearUserData() {
        configuration.clearUserData();
    }

    public String getPbAdSlot() {
        return configuration.getPbAdSlot();
    }

    public void setPbAdSlot(String pbAdSlot) {
        configuration.setPbAdSlot(pbAdSlot);
    }


    protected BidRequesterListener createBidListener(OnCompleteListener originalListener) {
        return new BidRequesterListener() {
            @Override
            public void onFetchCompleted(BidResponse response) {
                bidResponse = response;
                HashMap<String, String> keywords = response.getTargeting();
                Util.apply(keywords, adObject);
                originalListener.onComplete(ResultCode.SUCCESS, null);
            }

            @Override
            public void onError(AdException exception) {
                bidResponse = null;
                Util.apply(null, adObject);
                originalListener.onComplete(convertToResultCode(exception), null);
            }
        };
    }

    protected ResultCode convertToResultCode(AdException renderingException) {
        FetchDemandResult fetchDemandResult = FetchDemandResult.parseErrorMessage(renderingException.getMessage());
        LogUtil.error("Prebid", "Can't download bids: " + fetchDemandResult);
        switch (fetchDemandResult) {
            case INVALID_ACCOUNT_ID:
                return ResultCode.INVALID_ACCOUNT_ID;
            case INVALID_CONFIG_ID:
                return ResultCode.INVALID_CONFIG_ID;
            case INVALID_SIZE:
                return ResultCode.INVALID_SIZE;
            case INVALID_CONTEXT:
                return ResultCode.INVALID_CONTEXT;
            case INVALID_AD_OBJECT:
                return ResultCode.INVALID_AD_OBJECT;
            case INVALID_HOST_URL:
                return ResultCode.INVALID_HOST_URL;
            case NETWORK_ERROR:
                return ResultCode.NETWORK_ERROR;
            case TIMEOUT:
                return ResultCode.TIMEOUT;
            case NO_BIDS:
                return ResultCode.NO_BIDS;
            default:
                return ResultCode.PREBID_SERVER_ERROR;
        }
    }


    @VisibleForTesting
    public AdUnitConfiguration getConfiguration() {
        return configuration;
    }

}

