/*
 *    Copyright 2018-2021 Prebid.org, Inc.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.prebid.mobile.rendering.video;

import android.content.Context;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.annotation.OptIn;
import androidx.annotation.VisibleForTesting;

import androidx.media3.common.MediaItem;
import androidx.media3.common.PlaybackException;
import androidx.media3.common.Player;
import androidx.media3.common.util.UnstableApi;
import androidx.media3.common.util.Util;
import androidx.media3.datasource.DefaultDataSourceFactory;
import androidx.media3.exoplayer.ExoPlayer;
import androidx.media3.exoplayer.source.ProgressiveMediaSource;
import androidx.media3.ui.PlayerView;

import org.prebid.mobile.LogUtil;
import org.prebid.mobile.api.exceptions.AdException;
import org.prebid.mobile.api.rendering.VideoView;
import org.prebid.mobile.api.rendering.VisibilityTracker;
import org.prebid.mobile.configuration.AdUnitConfiguration;
import org.prebid.mobile.rendering.listeners.VideoCreativeViewListener;
import org.prebid.mobile.rendering.models.AdPosition;
import org.prebid.mobile.rendering.models.CreativeModel;
import org.prebid.mobile.rendering.video.vast.VASTErrorCodes;

import kotlin.Unit;
import kotlin.jvm.functions.Function2;

public class ExoPlayerView extends PlayerView implements VideoPlayerView {

    private static final String TAG = "ExoPlayerView";
    public static final float DEFAULT_INITIAL_VIDEO_VOLUME = 1.0f;

    @NonNull private final VideoCreativeViewListener videoCreativeViewListener;
    private AdViewProgressUpdateTask adViewProgressUpdateTask;
    private ExoPlayer player;

    private Uri videoUri;

    private long vastVideoDuration = -1;

    private Boolean playable = null;

    private VisibilityTracker visibilityTracker = new VisibilityTracker(this, new Handler(Looper.getMainLooper()), new Function2<Long, Long, Unit>() {
        @Override
        public Unit invoke(Long visibilePixels, Long totalPixels) {
            boolean playable = false;
            if (totalPixels > 0 && visibilePixels  * 100 / totalPixels > 95) {
                playable = true;
            }
            updatePlayableState(playable);
            return null;
        }
    });

    private boolean isInterstitial() {
        if (videoCreativeViewListener instanceof VideoCreative) {
           return ((VideoCreative) videoCreativeViewListener).isInterstitial();
        }
        return false;
    }

    private void updatePlayableState(boolean playable) {
        LogUtil.debug(TAG, "updatePlayableState. playable: " + playable);
        if (this.playable == null || this.playable != playable) {
            LogUtil.debug(TAG, "updatePlayableState. statue changed from: " + this.playable + " to: " + playable);
            this.playable = playable;
            if (this.playable) {
                new Handler(Looper.getMainLooper()).postDelayed(() -> {
                    if (!isPlaying()) {
                        resume();
                    }
                }, 200);
            } else {
                new Handler(Looper.getMainLooper()).postDelayed(() -> {
                    if (isPlaying()) {
                        pause();
                    }
                }, 200);
            }
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        if (isInterstitial()) {
            visibilityTracker.startTracking();
        }
    }

    @Override protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (isInterstitial()) {
            visibilityTracker.stopTracking();
        }
    }

    public ExoPlayerView(
            Context context,
            @NonNull VideoCreativeViewListener videoCreativeViewListener
    ) {
        super(context);
        this.videoCreativeViewListener = videoCreativeViewListener;
    }

    private final Player.Listener eventListener = new Player.Listener() {

        @Override
        public void onPlayerError(PlaybackException error) {
            videoCreativeViewListener.onFailure(new AdException(
                    AdException.INTERNAL_ERROR,
                    VASTErrorCodes.MEDIA_DISPLAY_ERROR.toString()
            ));
        }

        @Override
        public void onPlaybackStateChanged(int playbackState) {
            if (player == null) {
                LogUtil.debug(TAG, "onPlayerStateChanged(): Skipping state handling. Player is null");
                return;
            }
            switch (playbackState) {
                case Player.STATE_READY:
                    player.setPlayWhenReady(true);
                    if (!isInterstitial()) {
                        // Do not auto-release the player when immersive ads video duration reached.
                        initUpdateTask();
                    }
                    break;
                case Player.STATE_ENDED:
                    videoCreativeViewListener.onDisplayCompleted();
                    break;
            }
        }
    };

    @Override
    public void mute() {
        setVolume(0);
    }

    @Override
    public boolean isPlaying() {
        return player != null && player.getPlayWhenReady();
    }

    @Override
    public void unMute() {
        setVolume(DEFAULT_INITIAL_VIDEO_VOLUME);
    }

    @Override
    public void start(float initialVolume) {
        LogUtil.debug(TAG, "start() called");
        initLayout();
        initPlayer(initialVolume);
        preparePlayer(true);
        trackInitialStartEvent();
    }

    @Override
    public void setVastVideoDuration(long duration) {
        vastVideoDuration = duration;
    }

    @Override
    public long getCurrentPosition() {
        if (player == null) {
            return -1;
        }
        return player.getContentPosition();
    }

    @Override
    public void setVideoUri(Uri uri) {
        videoUri = uri;
    }

    @Override
    public int getDuration() {
        return (int) player.getDuration();
    }

    @Override
    public float getVolume() {
        return player.getVolume();
    }

    @Override
    public void resume() {
        LogUtil.debug(TAG, "resume() called");
        if (!isInterstitial()) {
            preparePlayer(false);
        } else {
            if (playable != null && !playable) {
                LogUtil.debug(TAG, "playable is false. skip resume()");
                return;
            }
            if (player != null) {
                player.play();
            }
        }
        videoCreativeViewListener.onEvent(VideoAdEvent.Event.AD_RESUME);
    }

    @Override
    public void pause() {
        LogUtil.debug(TAG, "pause() called");
        if (player != null) {
            if (!isInterstitial()) {
                player.stop();
            } else {
                player.pause();
            }
            videoCreativeViewListener.onEvent(VideoAdEvent.Event.AD_PAUSE);
        }
    }

    @Override
    public void forceStop() {
        destroy();
        videoCreativeViewListener.onDisplayCompleted();
    }

    @Override
    public void destroy() {
        LogUtil.debug(TAG, "destroy() called");
        killUpdateTask();
        if (player != null) {
            player.stop();
            player.removeListener(eventListener);
            setPlayer(null);
            player.release();
            player = null;
        }
    }

    @VisibleForTesting
    void setVolume(float volume) {
        if (player != null && volume >= 0.0f) {
            videoCreativeViewListener.onVolumeChanged(volume);
            player.setVolume(volume);
        }
    }

    @Override
    public void stop() {
        if (player != null) {
            player.stop();
            player.clearMediaItems();
        }
    }

    private void initLayout() {
        RelativeLayout.LayoutParams playerLayoutParams = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.MATCH_PARENT);
        playerLayoutParams.addRule(RelativeLayout.CENTER_IN_PARENT);
        setLayoutParams(playerLayoutParams);
    }

    private void initPlayer(float initialVolume) {
        if (player != null) {
            LogUtil.debug(TAG, "Skipping initPlayer(): Player is already initialized.");
            return;
        }
        player = new ExoPlayer.Builder(getContext()).build();
        if (isInterstitial()) {
            player.setRepeatMode(Player.REPEAT_MODE_ALL);
        }
        player.addListener(eventListener);
        setPlayer(this.player);
        setUseController(false);
        player.setVolume(initialVolume);
    }

    private void initUpdateTask() {
        if (adViewProgressUpdateTask != null) {
            LogUtil.debug(TAG, "initUpdateTask: AdViewProgressUpdateTask is already initialized. Skipping.");
            return;
        }

        try {
            adViewProgressUpdateTask = new AdViewProgressUpdateTask(
                    videoCreativeViewListener,
                    (int) player.getDuration()
            );
            adViewProgressUpdateTask.setVastVideoDuration(vastVideoDuration);
            adViewProgressUpdateTask.execute();
        }
        catch (AdException e) {
            e.printStackTrace();
        }
    }

    @OptIn(markerClass = UnstableApi.class) @VisibleForTesting
    void preparePlayer(boolean resetPosition) {
        ProgressiveMediaSource extractorMediaSource = buildMediaSource(videoUri);
        if (extractorMediaSource == null || player == null) {
            LogUtil.debug(TAG, "preparePlayer(): ExtractorMediaSource or ExoPlayer is null. Skipping prepare.");
            return;
        }
        player.setMediaSource(extractorMediaSource, resetPosition);
        player.prepare();
    }

    @OptIn(markerClass = UnstableApi.class) private ProgressiveMediaSource buildMediaSource(Uri uri) {
        if (uri == null) {
            return null;
        }
        MediaItem mediaItem = new MediaItem.Builder().setUri(uri).build();
        return new ProgressiveMediaSource.Factory(
                new DefaultDataSourceFactory(getContext(), Util.getUserAgent(getContext(), "PrebidRenderingSDK")))
                .createMediaSource(mediaItem);
    }

    private void killUpdateTask() {
        LogUtil.debug(TAG, "killUpdateTask() called");
        if (adViewProgressUpdateTask != null) {
            adViewProgressUpdateTask.cancel(true);
            adViewProgressUpdateTask = null;
        }
    }

    private void trackInitialStartEvent() {
        if (videoUri != null && player != null && player.getCurrentPosition() == 0) {
            videoCreativeViewListener.onEvent(VideoAdEvent.Event.AD_CREATIVEVIEW);
            videoCreativeViewListener.onEvent(VideoAdEvent.Event.AD_START);
        }
    }
}
