package org.prebid.mobile.api.rendering

import android.graphics.Rect
import android.os.Handler
import android.os.Looper
import android.view.View
import android.view.ViewTreeObserver

private const val VISIBILITY_THROTTLE_MILLIS = 100L

class VisibilityTracker(
    private val view: View,
    private val handler: Handler = Handler(Looper.getMainLooper()),
    private val onVisibilityChanged: (visiblePixels: Long, totalPixels: Long) -> Unit,
) {
    private var isTracking = false
    private var isVisibilityScheduled = false
    private val onPreDrawListener = ViewTreeObserver.OnPreDrawListener {
        if (isTracking) {
            scheduleVisibilityCheck()
        }
        true
    }

    fun startTracking() {
        if (!isTracking) {
            isTracking = true

            val observer = view.viewTreeObserver
            if (observer != null && observer.isAlive) {
                observer.addOnPreDrawListener(onPreDrawListener)
            }

            scheduleVisibilityCheck()
        }
    }

    fun stopTracking() {
        if (isTracking) {
            isTracking = false

            val observer = view.viewTreeObserver
            if (observer != null && observer.isAlive) {
                observer.removeOnPreDrawListener(onPreDrawListener)
            }

            handler.removeCallbacks(visibilityRunnable)
            isVisibilityScheduled = false
        }
    }

    private fun scheduleVisibilityCheck() {
        if (isVisibilityScheduled) {
            return
        }

        isVisibilityScheduled = true
        handler.postDelayed(visibilityRunnable, VISIBILITY_THROTTLE_MILLIS)
    }

    private val visibilityRunnable = object: Runnable {
        private val tempRect = Rect()
        private var lastVisiblePixels = 0L
        private var lastTotalPixels = 0L

        override fun run() {
            isVisibilityScheduled = false

            if (!view.isShown || view.windowVisibility != View.VISIBLE || !view.getGlobalVisibleRect(tempRect)) {
                tempRect.setEmpty()
            }

            val visiblePixels = tempRect.width().toLong() * tempRect.height()
            val totalPixels = view.width.toLong() * view.height

            if (visiblePixels != lastVisiblePixels || totalPixels != lastTotalPixels) {
                onVisibilityChanged(visiblePixels, totalPixels)
            }

            lastVisiblePixels = visiblePixels
            lastTotalPixels = totalPixels
        }
    }
}
