package com.particles.pubmaticadapter

import android.content.Context
import android.widget.Button
import android.widget.ImageView
import androidx.annotation.Keep
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.auction.BidderInfo
import com.particles.msp.util.Logger
import com.pubmatic.sdk.common.OpenWrapSDK
import com.pubmatic.sdk.common.OpenWrapSDKConfig
import com.pubmatic.sdk.common.OpenWrapSDKInitializer
import com.pubmatic.sdk.common.POBAdSize
import com.pubmatic.sdk.common.POBError
import com.pubmatic.sdk.common.models.POBApplicationInfo
import com.pubmatic.sdk.nativead.POBNativeAd
import com.pubmatic.sdk.nativead.POBNativeAdListener
import com.pubmatic.sdk.nativead.POBNativeAdLoader
import com.pubmatic.sdk.nativead.POBNativeAdLoaderListener
import com.pubmatic.sdk.nativead.datatype.POBNativeTemplateType
import com.pubmatic.sdk.nativead.views.POBNativeAdMediumTemplateView
import com.pubmatic.sdk.openwrap.banner.POBBannerView
import com.pubmatic.sdk.openwrap.core.POBBid
import com.pubmatic.sdk.openwrap.core.POBBidEvent
import com.pubmatic.sdk.openwrap.core.POBBidEventListener
import com.pubmatic.sdk.openwrap.interstitial.POBInterstitial
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import java.net.URL
import com.particles.msp.api.AdFormat as MspAdFormat
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView

@Keep
class PubMaticAdapter : AdNetworkAdapter() {
    private val tagPrefix = "[Adapter: PubMatic]"
    private var bannerAd: POBBannerView? = null
    private var nativeAd: POBNativeAd? = null
    private var nativeAdListener: POBNativeAdListener? = null
    private var interstitialAd: POBInterstitial? = null
    override val adNetworkName: String = "pubmatic"

    override fun destroyAd() {
        bannerAd?.destroy()
        bannerAd = null
        nativeAd?.destroy()
        nativeAd = null
        nativeAdListener = null
        interstitialAd?.destroy()
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context
    ) {
        super.initialize(initParams, adapterInitListener, context)
        val pubId = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PUBMATIC_PUBLISHER_ID] as? String
        val profileIds = (initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PUBMATIC_PROFILE_IDS] as? String)
            ?.split(',')?.mapNotNull { it.toIntOrNull() } ?: emptyList()
        if (pubId.isNullOrEmpty()) {
            Logger.info("$tagPrefix initialize failed: Publisher ID not found")
            adapterInitListener.onComplete(
                AdNetwork.PubMatic,
                AdapterInitStatus.FAILURE,
                AdapterInitStatus.FAILURE.message
            )
            return
        }

        publisherId = pubId

        CoroutineScope(Dispatchers.IO).launch {
            val config = OpenWrapSDKConfig.Builder(pubId, profileIds).build()
            OpenWrapSDK.initialize(context, config, object: OpenWrapSDKInitializer.Listener {
                override fun onFailure(error: POBError) {
                    Logger.info("PubMatic init failed: $error")
                    adapterInitListener.onComplete(
                        AdNetwork.PubMatic,
                        AdapterInitStatus.FAILURE,
                        error.errorMessage,
                    )
                }
                override fun onSuccess() {
                    adapterInitListener.onComplete(
                        AdNetwork.PubMatic,
                        AdapterInitStatus.SUCCESS,
                        AdapterInitStatus.SUCCESS.message,
                    )
                }
            })
            runCatching {
                URL("https://play.google.com/store/apps/details?id=${context.packageName}")
            }.getOrNull()?.let {
                OpenWrapSDK.setApplicationInfo(POBApplicationInfo().apply { storeURL = it })
            }
        }
    }

    override fun loadAdCreative(
        bidResponse: BidResponse?,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        bidderPlacementId: String,
        bidderInfo: BidderInfo
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        when (adRequest.adFormat) {
            MspAdFormat.BANNER -> loadBannerAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.NATIVE -> loadNativeAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.INTERSTITIAL -> loadInterstitialAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load pubmatic banner ads: $bidderPlacementId")

        val (publisherId, profileId, adUnitId) = runCatching {
            prepareToLoad(bidderPlacementId)
        }.getOrElse { e ->
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic native Ad: ${e.message}")
            return
        }

        val adSize = adRequest.adSize?.let {
            when {
                it.width == 320 && it.height == 50 -> POBAdSize.BANNER_SIZE_320x50
                it.width == 300 && it.height == 250 -> POBAdSize.BANNER_SIZE_300x250
                it.width == 728 && it.height == 90 -> POBAdSize.BANNER_SIZE_728x90
                it.width > 0 && it.height > 0 -> POBAdSize(it.width, it.height)
                else -> null
            }
        }

        if (adSize == null) {
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic Ad: invalid ad size.")
            return
        }

        val bannerAdListener = object : POBBannerView.POBBannerViewListener() {
            override fun onAdReceived(view: POBBannerView) {
                Logger.info("$tagPrefix successfully loaded pubmatic banner ads ...")
                val mspAd = MspBannerAdView(view, this@PubMaticAdapter)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, view.bid?.price ?: 0.0)
            }

            override fun onAdFailed(view: POBBannerView, error: POBError) {
                Logger.info("$tagPrefix failed to load pubmatic banner ads ...${error.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic banner Ad: ad not filled")
            }

            override fun onAdImpression(view: POBBannerView) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked(view: POBBannerView) {
                handleAdClicked(adListener, adRequest)
            }
        }

        val ad = POBBannerView(context, publisherId, profileId, adUnitId, adSize).also { bannerAd = it }
        ad.pauseAutoRefresh()
        ad.setListener(bannerAdListener)
        ad.loadAd()
    }

    private fun loadNativeAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load pubmatic native ads: $bidderPlacementId")

        val (publisherId, profileId, adUnitId) = runCatching {
            prepareToLoad(bidderPlacementId)
        }.getOrElse { e ->
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic native Ad: ${e.message}")
            return
        }

        val nativeAdListener = object : POBNativeAdListener {
            override fun onNativeAdRendered(nativeAd: POBNativeAd) {
            }

            override fun onNativeAdRenderingFailed(nativeAd: POBNativeAd, error: POBError) {
            }

            override fun onNativeAdImpression(nativeAd: POBNativeAd) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onNativeAdClicked(nativeAd: POBNativeAd) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onNativeAdClicked(nativeAd: POBNativeAd, assetId: String) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onNativeAdLeavingApplication(nativeAd: POBNativeAd) {
            }

            override fun onNativeAdOpened(nativeAd: POBNativeAd) {
            }

            override fun onNativeAdClosed(nativeAd: POBNativeAd) {
            }
        }

        val ad = POBNativeAdLoader(context, publisherId, profileId, adUnitId, POBNativeTemplateType.MEDIUM)
        val adLoaderListener = object : POBBidEventListener, POBNativeAdLoaderListener {
            private var price = 0.0

            override fun onBidReceived(bidEvent: POBBidEvent, bid: POBBid) {
                price = bid.price
                bidEvent.proceedToLoadAd()
            }

            override fun onBidFailed(bidEvent: POBBidEvent, error: POBError) {
                bidEvent.proceedToLoadAd()
            }

            override fun onAdReceived(adLoader: POBNativeAdLoader, nativeAd: POBNativeAd) {
                Logger.info("$tagPrefix successfully loaded pubmatic native ads ...")

                this@PubMaticAdapter.nativeAd = nativeAd
                this@PubMaticAdapter.nativeAdListener = nativeAdListener

                val mspAd = PubMaticNativeAd.Builder(this@PubMaticAdapter)
                    .mediaView(ImageView(context).apply {
                        scaleType = ImageView.ScaleType.FIT_CENTER
                    })
                    .build()
                    .apply {
//                        adInfo["isVideo"] = false
                    }

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, price)
            }

            override fun onFailedToLoad(adLoader: POBNativeAdLoader, error: POBError) {
                Logger.info("$tagPrefix failed to load pubmatic native ads ...${error.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic banner Ad: ${error.errorMessage}")
            }
        }
        ad.setBidEventListener(adLoaderListener)
        ad.setAdLoaderListener(adLoaderListener)
        ad.loadAd()
    }

    private fun loadInterstitialAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load pubmatic interstitial ads: $bidderPlacementId")

        val (publisherId, profileId, adUnitId) = runCatching {
            prepareToLoad(bidderPlacementId)
        }.getOrElse { e ->
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic native Ad: ${e.message}")
            return
        }

        val interstitialAdListener = object : POBInterstitial.POBInterstitialListener() {
            override fun onAdFailedToLoad(ad: POBInterstitial, error: POBError) {
                Logger.info("$tagPrefix failed to load pubmatic interstitial ads ...${error.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load PubMatic interstitial Ad: ad not filled")
            }

            override fun onAdReceived(ad: POBInterstitial) {
                Logger.info("$tagPrefix successfully loaded pubmatic interstitial ads ...")
                val mspAd = PubMaticInterstitialAd(this@PubMaticAdapter, ad)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, ad.bid?.price ?: 0.0)
            }

            override fun onAdImpression(ad: POBInterstitial) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked(ad: POBInterstitial) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdClosed(ad: POBInterstitial) {
                handleAdDismissed(adListener, adRequest)
            }
        }

        val ad = POBInterstitial(context, publisherId, profileId, adUnitId)
            .also { interstitialAd = it }
        ad.setListener(interstitialAdListener)
        ad.loadAd()
    }

    private fun prepareToLoad(bidderPlacementId: String): Triple<String, Int, String> {
        val pubId = publisherId
        if (pubId.isNullOrEmpty()) {
            throw IllegalArgumentException("invalid publisher id")
        }

        val (profileId, adUnitId) = bidderPlacementId.split('|', limit = 2)
            .run { first().toIntOrNull() to getOrNull(1) }

        if (profileId == null) {
            throw IllegalArgumentException("invalid profile id")
        }

        if (adUnitId.isNullOrEmpty()) {
            throw IllegalArgumentException("invalid ad unit id")
        }

        return Triple(pubId, profileId, adUnitId)
    }

    override fun prepareViewForInteraction(nativeAd: MspNativeAd, nativeAdView: Any) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val ad = this.nativeAd ?: return
        val nativeAdListener = this.nativeAdListener ?: return

        val adView = POBNativeAdMediumTemplateView(nativeAdView.context, null)
        Utils.moveChildren(nativeAdView, adView)
        adView.apply {
            iconImage = null
            title = nativeAdView.adTitleView
            description = nativeAdView.adBodyView
            cta = Button(context)
            mainImage = nativeAd.mediaView as? ImageView
        }
        ad.renderAd(adView, nativeAdListener)
        nativeAdView.AdCallToActionView?.text = adView.cta?.text
        nativeAdView.addView(adView)

        val clickableViews = listOfNotNull(
            nativeAdView.adTitleView,
            nativeAdView.adBodyView,
            nativeAdView.AdvertiserView,
            nativeAdView.AdCallToActionView,
            nativeAdView.AdMediaView,
        )
        ad.registerViewForInteraction(adView, clickableViews, nativeAdListener)
    }

    companion object {
        private var publisherId: String? = null
    }
}
