package com.particles.unityadapter

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.content.Context
import android.os.Bundle
import androidx.annotation.Keep
import com.ironsource.mediationsdk.IronSource
import com.ironsource.mediationsdk.ads.nativead.LevelPlayMediaView
import com.ironsource.mediationsdk.ads.nativead.LevelPlayNativeAd
import com.ironsource.mediationsdk.ads.nativead.LevelPlayNativeAdListener
import com.ironsource.mediationsdk.ads.nativead.NativeAdLayout
import com.ironsource.mediationsdk.adunit.adapter.utility.AdInfo
import com.ironsource.mediationsdk.logger.IronSourceError
import com.particles.mes.android.MesTracker
import com.particles.mes.android.MesTrackerExt.trackAdHide
import com.particles.mes.android.MesTrackerExt.trackAdImpression
import com.particles.mes.android.MesTrackerExt.trackAdReport
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.api.AdFormat as MspAdFormat
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger
import com.unity3d.mediation.LevelPlay
import com.unity3d.mediation.LevelPlayAdError
import com.unity3d.mediation.LevelPlayAdInfo
import com.unity3d.mediation.LevelPlayAdSize
import com.unity3d.mediation.LevelPlayConfiguration
import com.unity3d.mediation.LevelPlayInitError
import com.unity3d.mediation.LevelPlayInitListener
import com.unity3d.mediation.LevelPlayInitRequest
import com.unity3d.mediation.banner.LevelPlayBannerAdView
import com.unity3d.mediation.banner.LevelPlayBannerAdViewListener
import com.unity3d.mediation.interstitial.LevelPlayInterstitialAd
import com.unity3d.mediation.interstitial.LevelPlayInterstitialAdListener
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

@Keep
class UnityAdapter : AdNetworkAdapter() {
    private var nativeAd: LevelPlayNativeAd? = null
    private var bannerAd: LevelPlayBannerAdView? = null
    private var interstitialAd: LevelPlayInterstitialAd? = null

    override fun destroyAd() {
        bannerAd?.destroy()
        bannerAd = null
        interstitialAd = null
        nativeAd?.destroyAd()
        nativeAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Any?
    ) {
        if (context !is Context) return

        val appKey = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_UNITY_APP_KEY] as? String
        if (appKey.isNullOrEmpty()) {
            Logger.info("UnityAdapter initialize failed: App Key not found")
            adapterInitListener.onComplete(
                AdNetwork.Unity,
                AdapterInitStatus.FAILURE,
                AdapterInitStatus.FAILURE.message
            )
            return
        }

        mesTracker = initParams.getMesHostUrl().takeIf { it.isNotEmpty() }
            ?.let { MesTracker.obtain(it) }

        CoroutineScope(Dispatchers.IO).launch {
            setupIronSourceLifecycle(context)
            IronSource.setConsent(initParams.hasUserConsent())
            IronSource.setMetaData("do_not_sell", initParams.isDoNotSell().toString())
            IronSource.setMetaData("is_child_directed", initParams.isAgeRestrictedUser().toString())

            val initRequest = LevelPlayInitRequest.Builder(appKey)
//                .apply {
//                    UserId.getCachedUserId()?.let { withUserId(it.toString()) }
//                }
                .withLegacyAdFormats(listOf(
                    LevelPlay.AdFormat.NATIVE_AD,
                ))
                .build()

            LevelPlay.init(context, initRequest, object : LevelPlayInitListener {
                override fun onInitFailed(error: LevelPlayInitError) {
                    Logger.info("Unity init failed: $error")
                    adapterInitListener.onComplete(
                        AdNetwork.Unity,
                        AdapterInitStatus.FAILURE,
                        error.toString()
                    )
                }

                override fun onInitSuccess(configuration: LevelPlayConfiguration) {
                    adapterInitListener.onComplete(
                        AdNetwork.Unity,
                        AdapterInitStatus.SUCCESS,
                        AdapterInitStatus.SUCCESS.message
                    )
                }
            })
        }
    }

    private fun setupIronSourceLifecycle(context: Context) {
        val application = context.applicationContext as? Application
        application?.registerActivityLifecycleCallbacks(object : ActivityLifecycleCallbacks {
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {

            }

            override fun onActivityStarted(activity: Activity) {

            }

            override fun onActivityResumed(activity: Activity) {
                if (!isActivityExcluded(activity)) {
                    IronSource.onResume(activity)
                }
            }

            override fun onActivityPaused(activity: Activity) {
                if (!isActivityExcluded(activity)) {
                    IronSource.onPause(activity)
                }
            }

            override fun onActivityStopped(activity: Activity) {

            }

            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {

            }

            override fun onActivityDestroyed(activity: Activity) {

            }

            private fun isActivityExcluded(activity: Activity): Boolean {
                return listOf(
                    "com.google",
                    "com.facebook",
                    "com.particles.android.ads",
                    "org.prebid.mobile",
                    "com.ironsource",
                ).any { activity.javaClass.name.startsWith(it) }
            }
        })
    }

    override fun loadAdCreative(
        bidResponse: Any,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        bidderPlacementId: String
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId)

        when (adRequest.adFormat) {
            MspAdFormat.BANNER -> loadBannerAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.NATIVE -> loadNativeAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.INTERSTITIAL -> loadInterstitialAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            else -> adListener.onError("Failed to load Unity Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        val (adUnitId, placementName) = bidderPlacementId.split('|', limit = 2)
            .let { it[0] to it.getOrNull(1) }

        Logger.info("start to load unity banner ads: $bidderPlacementId")

        val ad = LevelPlayBannerAdView(context, adUnitId)
            .apply {
                setPlacementName(placementName)
            }

        val adSize = adRequest.adSize?.let {
            when {
                it.width == 320 && it.height == 50 -> LevelPlayAdSize.BANNER
                it.width == 320 && it.height == 90 -> LevelPlayAdSize.LARGE
                it.width == 300 && it.height == 250 -> LevelPlayAdSize.MEDIUM_RECTANGLE
                it.width > 0 && it.height > 0 -> LevelPlayAdSize.createCustomSize(it.width, it.height)
                else -> LevelPlayAdSize.createAdaptiveAdSize(context)
            }
        }
        adSize?.let { ad.setAdSize(adSize) }

        val bannerAdListener = object: LevelPlayBannerAdViewListener {
            override fun onAdLoadFailed(error: LevelPlayAdError) {
                Logger.info("failed to load unity banner ads ...${error.getErrorCode()}")
                adListener.onError("Failed to load Unity Ad: ${error.getErrorMessage()}")
            }

            override fun onAdLoaded(adInfo: LevelPlayAdInfo) {
                Logger.info("successfully loaded unity banner ads ...")
                ad.pauseAutoRefresh()
                val mspAd = MspBannerAdView(ad, this@UnityAdapter)
                    .apply {
                        this.adInfo["price"] = adInfo.getRevenue()
                        this.adInfo["bidder_placement_id"] = bidderPlacementId
                        this.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    }

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId)
            }

            override fun onAdDisplayed(adInfo: LevelPlayAdInfo) {
                super.onAdDisplayed(adInfo)
                handleAdImpression(adListener, adRequest)
                val mspAd = this@UnityAdapter.mspAd ?: return
                mesTracker?.trackAdImpression(adRequest, null, mspAd)
            }

            override fun onAdClicked(adInfo: LevelPlayAdInfo) {
                super.onAdClicked(adInfo)
                handleAdClicked(adListener, adRequest)
            }
        }
        ad.setBannerListener(bannerAdListener)
        ad.loadAd()
        bannerAd = ad
    }

    private fun loadNativeAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        val (adUnitId, placementName) = bidderPlacementId.split('|', limit = 2)
            .let { it[0] to it.getOrNull(1) }

        Logger.info("start to load unity native ads: $bidderPlacementId")

        val nativeAdListener = object : LevelPlayNativeAdListener {
            override fun onAdLoadFailed(nativeAd: LevelPlayNativeAd?, error: IronSourceError?) {
                Logger.info("failed to load unity native ads ...${error?.errorCode}")
                adListener.onError("Failed to load Unity native Ad: ${error?.errorMessage}")
            }

            override fun onAdLoaded(nativeAd: LevelPlayNativeAd?, adInfo: AdInfo?) {
                Logger.info("successfully loaded unity native ads ...")
                val ad = nativeAd ?: return

                val mspAd = UnityNativeAd.Builder(this@UnityAdapter)
                    .title(ad.title ?: "")
                    .body(ad.body ?: "")
                    .advertiser(ad.advertiser ?: "")
                    .callToAction(ad.callToAction ?: "")
                    .mediaView(LevelPlayMediaView(context))
                    .build()
                    .apply {
                        this.adInfo["price"] = adInfo?.revenue ?: 0L
                        this.adInfo["bidder_placement_id"] = bidderPlacementId
                        this.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    }

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId)
            }

            override fun onAdImpression(nativeAd: LevelPlayNativeAd?, adInfo: AdInfo?) {
                handleAdImpression(adListener, adRequest)
                val mspAd = this@UnityAdapter.mspAd ?: return
                mesTracker?.trackAdImpression(adRequest, null, mspAd)
            }

            override fun onAdClicked(nativeAd: LevelPlayNativeAd?, adInfo: AdInfo?) {
                handleAdClicked(adListener, adRequest)
            }
        }

        nativeAd = LevelPlayNativeAd.Builder()
            .withPlacementName(placementName)
            .withListener(nativeAdListener)
            .build()
            .also { it.loadAd() }
    }

    private fun loadInterstitialAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        val (adUnitId, placementName) = bidderPlacementId.split('|', limit = 2)
            .let { it[0] to it.getOrNull(1) }

        Logger.info("start to load unity interstitial ads: $bidderPlacementId")

        val ad = LevelPlayInterstitialAd(adUnitId)
        ad.setListener(object : LevelPlayInterstitialAdListener {
            override fun onAdLoadFailed(error: LevelPlayAdError) {
                Logger.info("failed to load unity interstitial ads ...${error.getErrorCode()}")
                adListener.onError("Failed to load Unity interstitial Ad: ${error.getErrorMessage()}")
            }

            override fun onAdLoaded(adInfo: LevelPlayAdInfo) {
                Logger.info("successfully loaded unity interstitial ads ...")
                val mspAd = UnityInterstitialAd(this@UnityAdapter, ad, placementName)
                    .apply {
                        this.adInfo["price"] = adInfo.getRevenue()
                        this.adInfo["bidder_placement_id"] = bidderPlacementId
                        this.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    }
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId)
            }

            override fun onAdDisplayed(adInfo: LevelPlayAdInfo) {
                handleAdImpression(adListener, adRequest)
                val mspAd = this@UnityAdapter.mspAd ?: return
                mesTracker?.trackAdImpression(adRequest, null, mspAd)
            }

            override fun onAdClicked(adInfo: LevelPlayAdInfo) {
                super.onAdClicked(adInfo)
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdClosed(adInfo: LevelPlayAdInfo) {
                super.onAdClosed(adInfo)
                handleAdDismissed(adListener, adRequest)
            }
        })
        ad.loadAd()
        interstitialAd = ad
    }

    override fun prepareViewForInteraction(nativeAd: MspNativeAd, nativeAdView: Any) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val adView = NativeAdLayout(nativeAdView.context)
            .apply {
                setAdvertiserView(nativeAdView.AdvertiserView)
                setTitleView(nativeAdView.adTitleView)
                setBodyView(nativeAdView.adBodyView)
                setMediaView(nativeAd.mediaView as? LevelPlayMediaView)
                setCallToActionView(nativeAdView.AdCallToActionView)
            }
        Utils.moveChildren(nativeAdView, adView)
        nativeAdView.addView(adView)
        this.nativeAd?.let { adView.registerNativeAdViews(it) }
    }

    override fun sendHideAdEvent(reason: String) {
        mspAd?.let { mesTracker?.trackAdHide(adRequest, null, it, reason)}
    }

    override fun sendReportAdEvent(reason: String, description: String?) {
        mspAd?.let { mesTracker?.trackAdReport(adRequest, null, it, reason, description)}
    }

    companion object {
        private var mesTracker: MesTracker? = null
    }
}
