/*
 * Copyright (C) 2017/2020 e-voyageurs technologies
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.tock.bot.admin

import ai.tock.bot.admin.answer.AnswerConfiguration
import ai.tock.bot.admin.answer.AnswerConfigurationType.builtin
import ai.tock.bot.admin.answer.AnswerConfigurationType.script
import ai.tock.bot.admin.answer.BuiltInAnswerConfiguration
import ai.tock.bot.admin.answer.ScriptAnswerConfiguration
import ai.tock.bot.admin.answer.ScriptAnswerVersionedConfiguration
import ai.tock.bot.admin.answer.SimpleAnswerConfiguration
import ai.tock.bot.admin.bot.BotApplicationConfiguration
import ai.tock.bot.admin.bot.BotApplicationConfigurationDAO
import ai.tock.bot.admin.bot.BotConfiguration
import ai.tock.bot.admin.bot.BotVersion
import ai.tock.bot.admin.dialog.ApplicationDialogFlowData
import ai.tock.bot.admin.dialog.DialogFlowTransitionStatsData
import ai.tock.bot.admin.dialog.DialogReportDAO
import ai.tock.bot.admin.dialog.DialogReportQueryResult
import ai.tock.bot.admin.kotlin.compiler.KotlinFile
import ai.tock.bot.admin.kotlin.compiler.client.KotlinCompilerClient
import ai.tock.bot.admin.model.BotAnswerConfiguration
import ai.tock.bot.admin.model.BotBuiltinAnswerConfiguration
import ai.tock.bot.admin.model.BotScriptAnswerConfiguration
import ai.tock.bot.admin.model.BotSimpleAnswerConfiguration
import ai.tock.bot.admin.model.BotStoryDefinitionConfiguration
import ai.tock.bot.admin.model.BotStoryDefinitionConfigurationMandatoryEntity
import ai.tock.bot.admin.model.BotStoryDefinitionConfigurationStep
import ai.tock.bot.admin.model.CreateI18nLabelRequest
import ai.tock.bot.admin.model.CreateStoryRequest
import ai.tock.bot.admin.model.DialogFlowRequest
import ai.tock.bot.admin.model.DialogsSearchQuery
import ai.tock.bot.admin.model.Feature
import ai.tock.bot.admin.model.StorySearchRequest
import ai.tock.bot.admin.model.UserSearchQuery
import ai.tock.bot.admin.model.UserSearchQueryResult
import ai.tock.bot.admin.story.StoryDefinitionConfiguration
import ai.tock.bot.admin.story.StoryDefinitionConfigurationDAO
import ai.tock.bot.admin.story.StoryDefinitionConfigurationMandatoryEntity
import ai.tock.bot.admin.story.StoryDefinitionConfigurationStep
import ai.tock.bot.admin.story.StoryDefinitionConfigurationSummary
import ai.tock.bot.admin.story.dump.ScriptAnswerVersionedConfigurationDump
import ai.tock.bot.admin.story.dump.StoryDefinitionConfigurationDump
import ai.tock.bot.admin.story.dump.StoryDefinitionConfigurationDumpController
import ai.tock.bot.admin.story.dump.StoryDefinitionConfigurationFeatureDump
import ai.tock.bot.admin.user.UserAnalytics
import ai.tock.bot.admin.user.UserAnalyticsQueryResult
import ai.tock.bot.admin.user.UserReportDAO
import ai.tock.bot.connector.ConnectorType
import ai.tock.bot.connector.ConnectorTypeConfiguration.Companion.connectorConfigurations
import ai.tock.bot.definition.IntentWithoutNamespace
import ai.tock.bot.engine.dialog.DialogFlowDAO
import ai.tock.bot.engine.feature.FeatureDAO
import ai.tock.bot.engine.feature.FeatureState
import ai.tock.nlp.admin.AdminService
import ai.tock.nlp.front.client.FrontClient
import ai.tock.nlp.front.service.applicationDAO
import ai.tock.nlp.front.shared.config.ApplicationDefinition
import ai.tock.nlp.front.shared.config.Classification
import ai.tock.nlp.front.shared.config.ClassifiedSentence
import ai.tock.nlp.front.shared.config.ClassifiedSentenceStatus.model
import ai.tock.nlp.front.shared.config.ClassifiedSentenceStatus.validated
import ai.tock.nlp.front.shared.config.EntityDefinition
import ai.tock.nlp.front.shared.config.EntityTypeDefinition
import ai.tock.nlp.front.shared.config.IntentDefinition
import ai.tock.nlp.front.shared.config.SentencesQuery
import ai.tock.shared.Dice
import ai.tock.shared.defaultLocale
import ai.tock.shared.defaultZoneId
import ai.tock.shared.injector
import ai.tock.shared.provide
import ai.tock.shared.security.UserLogin
import ai.tock.shared.vertx.WebVerticle.Companion.badRequest
import ai.tock.translator.I18nKeyProvider
import ai.tock.translator.I18nLabel
import ai.tock.translator.I18nLabelValue
import ai.tock.translator.Translator
import com.github.salomonbrys.kodein.instance
import mu.KotlinLogging
import org.litote.kmongo.Id
import org.litote.kmongo.toId
import java.time.DayOfWeek
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZonedDateTime
import java.time.format.TextStyle
import java.util.Locale
import java.util.stream.LongStream
import java.util.stream.Stream
import kotlin.streams.toList

/**
 *
 */
object BotAdminService {

    private val logger = KotlinLogging.logger {}

    private val userReportDAO: UserReportDAO by injector.instance()
    internal val dialogReportDAO: DialogReportDAO by injector.instance()
    private val applicationConfigurationDAO: BotApplicationConfigurationDAO by injector.instance()
    private val storyDefinitionDAO: StoryDefinitionConfigurationDAO by injector.instance()
    private val featureDAO: FeatureDAO by injector.instance()
    private val dialogFlowDAO: DialogFlowDAO get() = injector.provide()
    private val front = FrontClient

    private class BotStoryDefinitionConfigurationDumpController(
        override val targetNamespace: String,
        override val botId: String,
        val story: StoryDefinitionConfigurationDump,
        val application: ApplicationDefinition,
        val mainLocale: Locale,
        val user: UserLogin
    ) : StoryDefinitionConfigurationDumpController {

        override fun keepFeature(feature: StoryDefinitionConfigurationFeatureDump): Boolean =
            feature.botApplicationConfigurationId == null
                    || getBotConfigurationById(feature.botApplicationConfigurationId!!)?.namespace == targetNamespace

        override fun buildScript(
            script: ScriptAnswerVersionedConfigurationDump,
            compile: Boolean
        ): ScriptAnswerVersionedConfiguration {
            return if (compile && !KotlinCompilerClient.compilerDisabled) {
                val fileName = "T${Dice.newId()}.kt"
                val result = KotlinCompilerClient.compile(KotlinFile(script.script, fileName))
                if (result?.compilationResult == null) {
                    throw badRequest("Compilation error: ${result?.errors?.joinToString()}")
                } else {
                    val c = result.compilationResult!!
                    ScriptAnswerVersionedConfiguration(
                        script = script.script,
                        compiledCode = c.files.map { it.key.substring(0, it.key.length - ".class".length) to it.value },
                        version = BotVersion.getCurrentBotVersion(botId),
                        mainClassName = c.mainClass
                    )
                }
            } else {
                ScriptAnswerVersionedConfiguration(
                    script = script.script,
                    compiledCode = emptyList(),
                    version = script.version,
                    mainClassName = "",
                    date = script.date
                )
            }
        }

        override fun checkIntent(intent: IntentWithoutNamespace?): IntentWithoutNamespace? {
            if (intent != null) {
                createOrGetIntent(
                    targetNamespace,
                    intent.name,
                    application._id,
                    story.category
                )
                return intent
            }
            return null
        }
    }

    private fun createOrGetIntent(
        namespace: String,
        intentName: String,
        applicationId: Id<ApplicationDefinition>,
        intentCategory: String
    ): IntentDefinition =
        AdminService.createOrGetIntent(
            namespace,
            IntentDefinition(
                intentName,
                namespace,
                setOf(applicationId),
                emptySet(),
                category = intentCategory
            )
        )!!

    fun getBots(namespace: String, botId: String): List<BotConfiguration> {
        return applicationConfigurationDAO.getBotConfigurationsByNamespaceAndBotId(namespace, botId)
    }

    fun save(conf: BotConfiguration) {
        applicationConfigurationDAO.save(conf)
    }

    fun searchUsers(query: UserSearchQuery): UserSearchQueryResult {
        return UserSearchQueryResult(userReportDAO.search(query.toUserReportQuery()))
    }

    private fun formatHours(hours: List<String>): List<String> {
        return hours.map { "$it:00" }.toList()
    }

    private fun LocalDate.datesUntil(endExclusive: LocalDate): Stream<LocalDate> {
        val end = endExclusive.toEpochDay()
        val start: Long = toEpochDay()
        require(end >= start) { "$endExclusive < $this" }
        return LongStream.range(start, end).mapToObj { epochDay: Long ->
            LocalDate.ofEpochDay(epochDay)
        }
    }

    private fun getDatesBetween(startDate: LocalDate, endDate: LocalDate): List<String> {
        return startDate.datesUntil(endDate.plusDays(1))
            .map { it.toString() }.toList()
    }

    private fun buildHoursList(filterDate: LocalDate): List<String> {
        val now = LocalDateTime.now(defaultZoneId)
        val sameDay = now.dayOfMonth == filterDate.dayOfMonth
        return Array(24) { it }.filter { if (sameDay) it <= now.hour else true }.map { it.toString() }.toList()
    }

    private fun isSameDay(from: LocalDateTime?, to: LocalDateTime?): Boolean {
        return from != null && from.toLocalDate().isEqual(to?.toLocalDate())
    }

    fun searchUsersAnalytics(query: UserSearchQuery): UserAnalyticsQueryResult {
        val configurations = getBotConfigurationsByNamespaceAndNlpModel(query.namespace, query.applicationName)
        val connectorTypesId =
            connectorConfigurations.asSequence().toList().filter { it.connectorType != ConnectorType.rest }
                .map { it.connectorType.id }.plus("vsc")
        val confByTypes =
            configurations.filter { it.connectorType.id in connectorTypesId }.distinctBy { it.connectorType }
        val userAnalyticsQuery = query.toUserAnalyticsQuery()
        val groups = userReportDAO.search(userAnalyticsQuery)
            .groupBy {
                groupSelector(userAnalyticsQuery.from, userAnalyticsQuery.to, it.lastUserActionDateTime)
            }
        val (dates, usersByDate) = if (isSameDay(userAnalyticsQuery.from, userAnalyticsQuery.to)) {
            val hoursList = buildHoursList(userAnalyticsQuery.to.toLocalDate())
            val usersAnalytics = hoursList.map {
                if (groups[it] != null) {
                    groups[it]
                } else {
                    listOf()
                }
            }.toList()
            Pair(formatHours(hoursList), usersAnalytics)
        } else {
            val datesBetween =
                getDatesBetween(userAnalyticsQuery.from.toLocalDate(), userAnalyticsQuery.to.toLocalDate())
            val usersAnalytics = datesBetween.map {
                if (groups[it] != null) {
                    groups[it]
                } else {
                    listOf<UserAnalytics>()
                }
            }.toList()
            Pair(datesBetween, usersAnalytics)
        }
        val result = arrayListOf<List<Int?>>()
        usersByDate.forEach { users ->
            run {
                val messagesNumber = arrayListOf<Int?>()
                confByTypes.forEach { connector ->
                    run {
                        val connectorAppIds = getConnectorAppIds(connector.connectorType, configurations)
                        val count = users?.count { user -> connectorAppIds.contains(user.applicationIds.first()) }
                        messagesNumber.add(count)
                    }
                }
                result.add(messagesNumber)
            }
        }
        return UserAnalyticsQueryResult(dates, result, confByTypes.map { it.connectorType.id })
    }

    private fun <S> reportMessagesByDateAndSeries(
        request: DialogFlowRequest,
        applications: Set<BotApplicationConfiguration>,
        series: Set<S>,
        seriesFilter: (S) -> (DialogFlowTransitionStatsData) -> Boolean,
        seriesLabel: (S) -> String
    ): UserAnalyticsQueryResult {
        val namespace = request.namespace
        val botId = request.botId
        val applicationIds = applications.map { it._id }.toSet()
        val fromDate = request.from?.toLocalDateTime()
        val toDate = request.to?.toLocalDateTime()
        logger.debug { "Building 'Messages by Configuration' report for ${applications.size} configurations: $applicationIds..." }

        val queryResult = dialogFlowDAO.search(namespace, botId, applicationIds, request.from, request.to)
            .groupBy { groupSelector(fromDate, toDate, it.date) }

        val (dates, transitionsByDate) = sortMessagesByDate(fromDate, toDate, queryResult)
        val result = arrayListOf<List<Int?>>()
        transitionsByDate.forEach { transitions ->
            run {
                val datesMessages = arrayListOf<Int?>()
                series.forEach { serie ->
                    run {
                        val count = transitions?.count(seriesFilter(serie))
                        datesMessages.add(count)
                    }
                }
                result.add(datesMessages)
            }
        }
        return UserAnalyticsQueryResult(dates, result, series.map { seriesLabel(it) })
    }

    private fun reportMessagesByDateAndFunction(
        request: DialogFlowRequest,
        applications: Set<BotApplicationConfiguration>,
        searchFunction: (String, String, Set<Id<BotApplicationConfiguration>>, ZonedDateTime?, ZonedDateTime?) -> List<DialogFlowTransitionStatsData>,
        seriesLabel: (String?) -> String = { "$it" }
    )
            : UserAnalyticsQueryResult {
        val namespace = request.namespace
        val botId = request.botId
        val applicationIds = applications.map { it._id }.toSet()
        val fromDate = request.from?.toLocalDateTime()
        val toDate = request.to?.toLocalDateTime()
        logger.debug { "Building 'Messages by Configuration' report for ${applications.size} configurations: $applicationIds..." }

        val messages = (searchFunction)(namespace, botId, applicationIds, request.from, request.to)
        val series = messages.groupingBy { it.text }.eachCount().toList().sortedByDescending { it.second }.unzip().first
        val messagesByDate = messages
            .groupBy { groupSelector(fromDate, toDate, it.date) }

        val (dates, transitionsByDate) = sortMessagesByDate(fromDate, toDate, messagesByDate)
        val result = arrayListOf<List<Int?>>()
        transitionsByDate.forEach { transitions ->
            run {
                val datesMessages = arrayListOf<Int?>()
                series.forEach { serie ->
                    run {
                        val count = transitions?.count { it.text == serie }
                        datesMessages.add(count)
                    }
                }
                result.add(datesMessages)
            }
        }
        return UserAnalyticsQueryResult(dates, result, series.map(seriesLabel))
    }

    private fun <S> reportMessagesBySeries(
        request: DialogFlowRequest,
        applications: Set<BotApplicationConfiguration>,
        series: Set<S>,
        seriesFilter: (S) -> (DialogFlowTransitionStatsData) -> Boolean,
        seriesLabel: (S) -> String
    ): UserAnalyticsQueryResult {
        val namespace = request.namespace
        val botId = request.botId
        val applicationIds = applications.map { it._id }.toSet()
        logger.debug { "Building 'Messages by Configuration' report for ${applications.size} configurations: $applicationIds..." }

        val transitions = dialogFlowDAO.search(namespace, botId, applicationIds, request.from, request.to)

        val result = arrayListOf<List<Int?>>()
        run {
            val seriesMessages = arrayListOf<Int?>()
            series.forEach { serie ->
                run {
                    val count = transitions.count(seriesFilter(serie))
                    seriesMessages.add(count)
                }
            }
            result.add(seriesMessages)
        }
        return UserAnalyticsQueryResult(listOf("All Range"), result, series.map { seriesLabel(it) })
    }

    private fun reportMessagesByFunction(
        request: DialogFlowRequest,
        applications: Set<BotApplicationConfiguration>,
        searchFunction: (String, String, Set<Id<BotApplicationConfiguration>>, ZonedDateTime?, ZonedDateTime?) -> List<DialogFlowTransitionStatsData>,
        seriesLabel: (String?) -> String = { "$it" }
    ): UserAnalyticsQueryResult {
        val namespace = request.namespace
        val botId = request.botId
        val applicationIds = applications.map { it._id }.toSet()
        logger.debug { "Building 'Messages by Configuration' report for ${applications.size} configurations: $applicationIds..." }

        val messages = (searchFunction)(namespace, botId, applicationIds, request.from, request.to)
        val series: Set<String> = messages.groupBy { it.text }.keys.map { seriesLabel(it) }.toSet()

        val result = arrayListOf<List<Int?>>()
        run {
            val seriesMessages = arrayListOf<Int?>()
            series.forEach { serie ->
                run {
                    val count = messages.count { seriesLabel(it.text) == serie }
                    seriesMessages.add(count)
                }
            }
            result.add(seriesMessages)
        }
        return UserAnalyticsQueryResult(listOf("All Range"), result, series.toList())
    }

    fun reportMessagesByType(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByDateAndSeries(request, applications,
            setOf(false, request.includeTestConfigurations),
            { isTests ->
                { transition ->
                    applications.find { it._id.toString() == transition.applicationId }?.applicationId?.startsWith(
                        "test-"
                    ) == isTests
                }
            }, { if (it) "Tests" else "Users" })
    }

    fun reportMessagesByConnectorType(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByDateAndSeries(request,
            applications,
            applications.map { it.connectorType }.toSet(),
            { connectorType -> { transition -> applications.find { it._id.toString() == transition.applicationId }?.connectorType == connectorType } },
            { it.id })
    }

    fun reportMessagesByConfiguration(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByDateAndSeries(
            request,
            applications,
            applications,
            { application -> { transition -> application._id.toString() == transition.applicationId } },
            { it.applicationId })
    }

    fun reportMessagesByDayOfWeek(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesBySeries(request,
            applications,
            DayOfWeek.values().toSet(),
            { day -> { transition -> transition.date.dayOfWeek == day } },
            { it.getDisplayName(TextStyle.FULL_STANDALONE, defaultLocale) })
    }

    fun reportMessagesByHour(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesBySeries(request, applications, (0..24).toList().toSet(),
            { hour -> { transition -> transition.date.hour == hour } }, { "${it}h" })
    }

    fun reportMessagesByIntent(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByFunction(request, applications, dialogFlowDAO::searchByDateWithIntent)
    }

    fun reportMessagesByDateAndIntent(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByDateAndFunction(request, applications, dialogFlowDAO::searchByDateWithIntent)
    }

    fun reportMessagesByStory(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        val stories = storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(request.namespace, request.botId)
        return reportMessagesByFunction(
            request, applications, dialogFlowDAO::searchByDateWithStory
        ) { it?.let { stories.find { story -> story._id.toString() == it }?.name } ?: "$it" }
    }

    fun reportMessagesByDateAndStory(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        val stories = storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(request.namespace, request.botId)
        return reportMessagesByDateAndFunction(
            request, applications, dialogFlowDAO::searchByDateWithStory
        ) { it?.let { stories.find { story -> story._id.toString() == it }?.name } ?: "$it" }
    }

    fun reportMessagesByStoryType(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        val stories = storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(request.namespace, request.botId)
        return reportMessagesByFunction(
            request, applications, dialogFlowDAO::searchByDateWithStory
        ) { it?.let { stories.find { story -> story._id.toString() == it }?.currentType?.name } ?: "unknown" }
    }

    fun reportMessagesByStoryCategory(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        val stories = storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(request.namespace, request.botId)
        return reportMessagesByFunction(
            request, applications, dialogFlowDAO::searchByDateWithStory
        ) { it?.let { stories.find { story -> story._id.toString() == it }?.category } ?: "unknown" }
    }

    fun reportMessagesByStoryLocale(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        val stories = storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(request.namespace, request.botId)
        return reportMessagesByFunction(
            request, applications, dialogFlowDAO::searchByDateWithStory
        ) {
            it?.let { stories.find { story -> story._id.toString() == it }?.userSentenceLocale.toString() } ?: "unknown"
        }
    }

    fun reportMessagesByActionType(request: DialogFlowRequest): UserAnalyticsQueryResult {
        val applications = loadApplications(request)
        return reportMessagesByFunction(request, applications, dialogFlowDAO::searchByDateWithActionType)
    }

    private fun sortMessagesByDate(
        fromDate: LocalDateTime?,
        toDate: LocalDateTime?,
        queryResult: Map<String, List<DialogFlowTransitionStatsData>>
    ) =
        if (isSameDay(fromDate, toDate)) {
            val hoursList = buildHoursList(toDate!!.toLocalDate())
            val usersAnalytics = hoursList.map {
                if (queryResult[it] != null) {
                    queryResult[it]
                } else {
                    listOf()
                }
            }.toList()
            Pair(formatHours(hoursList), usersAnalytics)
        } else {
            val datesBetween = getDatesBetween(fromDate!!.toLocalDate(), toDate!!.toLocalDate())
            val usersAnalytics = datesBetween.map {
                if (queryResult[it] != null) {
                    queryResult[it]
                } else {
                    listOf<DialogFlowTransitionStatsData>()
                }
            }.toList()
            Pair(datesBetween, usersAnalytics)
        }

    private fun groupSelector(from: LocalDateTime?, to: LocalDateTime?, elementDate: LocalDateTime) =
        if (isSameDay(from, to)) elementDate.hour.toString() else elementDate.toLocalDate().toString()

    private fun getConnectorAppIds(
        connectorType: ConnectorType,
        configurations: List<BotApplicationConfiguration>
    ): List<String> {
        return configurations.filter { config -> config.connectorType == connectorType }
            .map { config -> config.applicationId }
    }

    fun search(query: DialogsSearchQuery): DialogReportQueryResult {
        return dialogReportDAO.search(query.toDialogReportQuery())
            .run {
                if (query.skipObfuscation) {
                    this
                } else {
                    copy(dialogs = dialogs.map { d ->
                        var obfuscatedDialog = false
                        val actions = d.actions.map {
                            val obfuscatedMessage = it.message.obfuscate()
                            obfuscatedDialog = obfuscatedDialog || it.message != obfuscatedMessage
                            it.copy(message = obfuscatedMessage)
                        }
                        d.copy(
                            actions = actions,
                            obfuscated = obfuscatedDialog
                        )
                    })
                }
            }
    }

    fun deleteApplicationConfiguration(conf: BotApplicationConfiguration) {
        applicationConfigurationDAO.delete(conf)
        //delete rest connector if found
        applicationConfigurationDAO.getConfigurationByTargetId(conf._id)
            ?.also { applicationConfigurationDAO.delete(it) }
    }

    fun getBotConfigurationById(id: Id<BotApplicationConfiguration>): BotApplicationConfiguration? {
        return applicationConfigurationDAO.getConfigurationById(id)
    }

    fun getBotConfigurationByApplicationIdAndBotId(
        namespace: String,
        applicationId: String,
        botId: String
    ): BotApplicationConfiguration? {
        return applicationConfigurationDAO.getConfigurationByApplicationIdAndBotId(namespace, applicationId, botId)
    }

    fun getBotConfigurationsByNamespaceAndBotId(namespace: String, botId: String): List<BotApplicationConfiguration> {
        return applicationConfigurationDAO.getConfigurationsByNamespaceAndBotId(namespace, botId)
    }

    fun getBotConfigurationsByNamespaceAndNlpModel(
        namespace: String,
        applicationName: String
    ): List<BotApplicationConfiguration> {
        val app = applicationDAO.getApplicationByNamespaceAndName(namespace, applicationName)
        return if (app == null) emptyList() else applicationConfigurationDAO.getConfigurationsByNamespaceAndNlpModel(
            namespace,
            app.name
        )
    }

    fun saveApplicationConfiguration(conf: BotApplicationConfiguration) {
        applicationConfigurationDAO.save(conf)
        if (applicationConfigurationDAO.getBotConfigurationsByNamespaceAndNameAndBotId(
                conf.namespace,
                conf.name,
                conf.botId
            ) == null
        ) {
            applicationConfigurationDAO.save(
                BotConfiguration(
                    conf.name,
                    conf.botId,
                    conf.namespace,
                    conf.nlpModel
                )
            )
        }
    }

    fun searchStories(request: StorySearchRequest): List<StoryDefinitionConfigurationSummary> =
        storyDefinitionDAO.searchStoryDefinitionSummaries(request.toSummaryRequest())

    fun loadStories(request: StorySearchRequest): List<BotStoryDefinitionConfiguration> =
        findStories(request.namespace, request.applicationName).map {
            BotStoryDefinitionConfiguration(it, request.currentLanguage)
        }

    private fun findStories(namespace: String, applicationName: String): List<StoryDefinitionConfiguration> {
        val botConf =
            getBotConfigurationsByNamespaceAndNlpModel(namespace, applicationName).firstOrNull()
        return if (botConf == null) {
            emptyList()
        } else {
            storyDefinitionDAO
                .getStoryDefinitionsByNamespaceAndBotId(namespace, botConf.botId)
        }
    }

    fun exportStories(namespace: String, applicationName: String): List<StoryDefinitionConfigurationDump> =
        findStories(namespace, applicationName).map { StoryDefinitionConfigurationDump(it) }

    fun exportStory(
        namespace: String,
        applicationName: String,
        storyDefinitionId: String
    ): StoryDefinitionConfigurationDump? {
        val botConf =
            getBotConfigurationsByNamespaceAndNlpModel(namespace, applicationName).firstOrNull()
        return if (botConf != null) {
            val story =
                storyDefinitionDAO.getStoryDefinitionsByNamespaceBotIdStoryId(
                    namespace = namespace,
                    botId = botConf.botId,
                    storyId = storyDefinitionId
                )
            story?.let { StoryDefinitionConfigurationDump(it) }
        } else null
    }

    fun findStory(namespace: String, storyDefinitionId: String): BotStoryDefinitionConfiguration? {
        val story = storyDefinitionDAO.getStoryDefinitionById(storyDefinitionId.toId())
        return loadStory(namespace, story)
    }

    fun findRuntimeStorySettings(namespace: String, botId: String): List<BotStoryDefinitionConfiguration>? {
        val stories = storyDefinitionDAO.getRuntimeStorySettings(namespace, botId)
        return stories.mapNotNull { story -> loadStory(namespace, story) }
    }

    private fun loadStory(namespace: String, conf: StoryDefinitionConfiguration?): BotStoryDefinitionConfiguration? {
        if (conf?.namespace == namespace) {
            val botConf = getBotConfigurationsByNamespaceAndBotId(namespace, conf.botId).firstOrNull()
            if (botConf != null) {
                val applicationDefinition = applicationDAO.getApplicationByNamespaceAndName(namespace, botConf.nlpModel)
                return BotStoryDefinitionConfiguration(
                    conf,
                    applicationDefinition?.supportedLocales?.firstOrNull() ?: defaultLocale
                )
            }
        }
        return null
    }

    fun importStories(
        namespace: String,
        botId: String,
        locale: Locale,
        stories: List<StoryDefinitionConfigurationDump>,
        user: UserLogin
    ) {
        val botConf = getBotConfigurationsByNamespaceAndBotId(namespace, botId).firstOrNull()

        if (botConf == null) {
            badRequest("No bot configuration is defined yet")
        } else {
            val application = front.getApplicationByNamespaceAndName(namespace, botConf.nlpModel)!!
            stories.forEach {
                val controller =
                    BotStoryDefinitionConfigurationDumpController(namespace, botId, it, application, locale, user)
                val storyConf = it.toStoryDefinitionConfiguration(controller)
                importStory(namespace, storyConf, botConf, controller)
            }
        }
    }

    private fun importStory(
        namespace: String,
        story: StoryDefinitionConfiguration,
        botConf: BotApplicationConfiguration,
        controller: BotStoryDefinitionConfigurationDumpController
    ) {

        storyDefinitionDAO.getStoryDefinitionByNamespaceAndBotIdAndIntent(
            namespace,
            botConf.botId,
            story.intent.name
        )?.also {
            storyDefinitionDAO.delete(it)
        }
        storyDefinitionDAO.getStoryDefinitionByNamespaceAndBotIdAndStoryId(
            namespace,
            botConf.botId,
            story.storyId
        )?.also {
            storyDefinitionDAO.delete(it)
        }

        storyDefinitionDAO.save(story)

        val mainIntent = createOrGetIntent(
            namespace,
            story.intent.name,
            controller.application._id,
            story.category
        )
        if (story.userSentence.isNotBlank()) {
            saveSentence(
                story.userSentence,
                story.userSentenceLocale ?: controller.mainLocale,
                controller.application._id,
                mainIntent._id,
                controller.user
            )
        }

        //save all intents of steps
        story.steps.forEach { saveUserSentenceOfStep(controller.application, it, controller.user) }
    }

    fun findConfiguredStoryByBotIdAndIntent(
        namespace: String,
        botId: String,
        intent: String
    ): BotStoryDefinitionConfiguration? {
        return storyDefinitionDAO.getConfiguredStoryDefinitionByNamespaceAndBotIdAndIntent(namespace, botId, intent)
            ?.let {
                loadStory(namespace, it)
            }
    }

    fun deleteStory(namespace: String, storyDefinitionId: String): Boolean {
        val story = storyDefinitionDAO.getStoryDefinitionById(storyDefinitionId.toId())
        if (story != null) {
            val botConf = getBotConfigurationsByNamespaceAndBotId(namespace, story.botId).firstOrNull()
            if (botConf != null) {
                storyDefinitionDAO.delete(story)
            }
        }
        return false
    }

    fun createStory(
        namespace: String,
        request: CreateStoryRequest,
        user: UserLogin
    ): IntentDefinition? {

        val botConf =
            getBotConfigurationsByNamespaceAndBotId(namespace, request.story.botId).firstOrNull()
        return if (botConf != null) {
            val nlpApplication = front.getApplicationByNamespaceAndName(namespace, botConf.nlpModel)!!
            val intentDefinition =
                createOrGetIntent(
                    namespace,
                    request.story.intent.name,
                    nlpApplication._id,
                    request.story.category
                )

            //create the story
            saveStory(namespace, request.story, user)
            request.firstSentences.filter { it.isNotBlank() }.forEach {
                saveSentence(it, request.language, nlpApplication._id, intentDefinition._id, user)
            }
            intentDefinition
        } else {
            null
        }
    }

    private fun simpleAnswer(
        answer: BotSimpleAnswerConfiguration
    ): SimpleAnswerConfiguration {

        return SimpleAnswerConfiguration(
            answer.answers.map { it.toConfiguration() }
        )
    }

    private fun scriptAnswer(
        botId: String,
        oldAnswer: ScriptAnswerConfiguration?,
        answer: BotScriptAnswerConfiguration
    ): ScriptAnswerConfiguration? {

        val script = answer.current.script
        if (!KotlinCompilerClient.compilerDisabled && oldAnswer?.current?.script != script) {
            val fileName = "T${Dice.newId()}.kt"
            val result = KotlinCompilerClient.compile(KotlinFile(script, fileName))
            if (result?.compilationResult == null) {
                throw badRequest("Compilation error: ${result?.errors?.joinToString()}")
            } else {
                val c = result.compilationResult!!
                val newScript = ScriptAnswerVersionedConfiguration(
                    script,
                    c.files.map { it.key.substring(0, it.key.length - ".class".length) to it.value },
                    BotVersion.getCurrentBotVersion(botId),
                    c.mainClass
                )
                return ScriptAnswerConfiguration(
                    (oldAnswer?.scriptVersions ?: emptyList()) + newScript,
                    newScript
                )
            }
        } else {
            return oldAnswer
        }
    }

    private fun BotAnswerConfiguration.toConfiguration(
        botId: String,
        answers: List<AnswerConfiguration>?
    ): AnswerConfiguration? =
        when (this) {
            is BotSimpleAnswerConfiguration -> simpleAnswer(this)
            is BotScriptAnswerConfiguration ->
                scriptAnswer(
                    botId,
                    answers?.find { it.answerType == script } as? ScriptAnswerConfiguration,
                    this
                )
            is BotBuiltinAnswerConfiguration -> BuiltInAnswerConfiguration(storyHandlerClassName)
            else -> error("unsupported type $this")
        }

    private fun BotAnswerConfiguration.toStoryConfiguration(
        botId: String,
        oldStory: StoryDefinitionConfiguration?
    ): AnswerConfiguration? =
        toConfiguration(botId, oldStory?.answers)

    private fun BotStoryDefinitionConfigurationMandatoryEntity.toEntityConfiguration(
        app: ApplicationDefinition,
        botId: String,
        oldStory: StoryDefinitionConfiguration?
    ): StoryDefinitionConfigurationMandatoryEntity =
        StoryDefinitionConfigurationMandatoryEntity(
            role,
            entityType,
            intent,
            answers.mapNotNull { botAnswerConfiguration ->
                botAnswerConfiguration.toConfiguration(
                    botId,
                    oldStory?.mandatoryEntities?.find { it.role == role }?.answers
                )
            },
            currentType
        ).apply {
            //if entity is null, it means that entity has not been modified
            if (entity != null) {
                //check that the intent & entity exist
                var newIntent = front.getIntentByNamespaceAndName(app.namespace, intent.name)
                val existingEntity = newIntent?.findEntity(role)
                val entityTypeName = entity.entityTypeName
                if (existingEntity == null) {
                    if (front.getEntityTypeByName(entityTypeName) == null) {
                        front.save(EntityTypeDefinition(entityTypeName))
                    }
                }
                if (newIntent == null) {
                    newIntent = IntentDefinition(
                        intent.name,
                        app.namespace,
                        setOf(app._id),
                        setOf(EntityDefinition(entityTypeName, role)),
                        label = intentDefinition?.label,
                        category = intentDefinition?.category,
                        description = intentDefinition?.description
                    )
                    front.save(newIntent)
                } else if (existingEntity == null) {
                    front.save(
                        newIntent.copy(
                            applications = newIntent.applications + app._id,
                            entities = newIntent.entities + EntityDefinition(entityTypeName, role)
                        )
                    )
                }
            }
        }

    private fun BotStoryDefinitionConfigurationStep.toStepConfiguration(
        app: ApplicationDefinition,
        botId: String,
        oldStory: StoryDefinitionConfiguration?
    ): StoryDefinitionConfigurationStep =
        StoryDefinitionConfigurationStep(
            name.takeUnless { it.isBlank() }
                ?: "${intent?.name}${(entity?.value ?: entity?.entityRole)?.let { "_$it" }}_$level",
            intent?.takeIf { it.name.isNotBlank() },
            targetIntent?.takeIf { it.name.isNotBlank() },
            answers.mapNotNull { botAnswerConfiguration ->
                botAnswerConfiguration.toConfiguration(
                    botId,
                    oldStory?.steps?.find { it.name == name }?.answers
                )
            },
            currentType,
            userSentence.defaultLabel ?: "",
            I18nLabelValue(userSentence),
            children.map { it.toStepConfiguration(app, botId, oldStory) },
            level,
            entity
        ).apply {
            updateIntentDefinition(intentDefinition, intent, app)
            updateIntentDefinition(targetIntentDefinition, targetIntent, app)
        }

    private fun updateIntentDefinition(
        intentDefinition: IntentDefinition?,
        intent: IntentWithoutNamespace?,
        app: ApplicationDefinition
    ) {
        //if intentDefinition is null, we don't need to update intent
        if (intentDefinition != null) {
            //check that the intent exists
            val intentName = intent?.name
            var newIntent = intentName?.let { front.getIntentByNamespaceAndName(app.namespace, intentName) }
            if (newIntent == null && intentName != null) {
                newIntent = IntentDefinition(
                    intentName,
                    app.namespace,
                    setOf(app._id),
                    emptySet(),
                    label = intentDefinition.label,
                    category = intentDefinition.category,
                    description = intentDefinition.description
                )
                front.save(newIntent)
            }
        }
    }

    private fun mergeStory(
        oldStory: StoryDefinitionConfiguration,
        story: BotStoryDefinitionConfiguration,
        application: ApplicationDefinition,
        botId: String
    ): StoryDefinitionConfiguration {
        return oldStory.copy(
            name = story.name,
            description = story.description,
            category = story.category,
            currentType = story.currentType,
            intent = story.intent,
            answers = story.answers.mapNotNull { it.toStoryConfiguration(botId, oldStory) },
            mandatoryEntities = story.mandatoryEntities.map {
                it.toEntityConfiguration(
                    application,
                    botId,
                    oldStory
                )
            },
            steps = story.steps.map { it.toStepConfiguration(application, botId, oldStory) },
            userSentence = story.userSentence,
            userSentenceLocale = story.userSentenceLocale,
            configurationName = story.configurationName,
            features = story.features,
            tags = story.tags
        )
    }

    fun saveStory(
        namespace: String,
        story: BotStoryDefinitionConfiguration,
        user: UserLogin
    ): BotStoryDefinitionConfiguration? {

        // Two stories (built-in or configured) should not have the same _id
        // There should be max one built-in (resp. configured) story for given namespace+bot+intent (or namespace+bot+storyId)
        // It can be updated if storyId remains the same, fails otherwise
        // Only a built-in story can change its type (to "manage it")

        val storyWithSameId = storyDefinitionDAO.getStoryDefinitionById(story._id)
        storyWithSameId?.let {
            val existingType = it.currentType
            if (existingType != story.currentType && existingType != builtin) {
                badRequest("Story ${it.name} ($existingType) already exists with the ID")
            }
        }

        val botConf = getBotConfigurationsByNamespaceAndBotId(namespace, story.botId).firstOrNull()
        return if (botConf != null) {

            val application = front.getApplicationByNamespaceAndName(namespace, botConf.nlpModel)!!
            val storyWithSameNsBotAndName =
                storyDefinitionDAO.getStoryDefinitionByNamespaceAndBotIdAndStoryId(
                    namespace,
                    botConf.botId,
                    story.storyId
                )?.also { logger.debug { "Found story with same namespace, type and name: $it" } }
            val storyWithSameNsBotAndIntent =
                storyDefinitionDAO.getStoryDefinitionByNamespaceAndBotIdAndIntent(
                    namespace,
                    botConf.botId,
                    story.intent.name
                )?.also { logger.debug { "Found story with same namespace, type and intent: $it" } }

            storyWithSameNsBotAndIntent.let {
                if (it == null || it.currentType == builtin) {
                    //intent change
                    if (storyWithSameId?._id != null) {
                        createOrGetIntent(
                            namespace,
                            story.intent.name,
                            application._id,
                            story.category
                        )
                    }
                } else {
                    if (story._id != it._id) {
                        badRequest("Story ${it.name} (${it.currentType}) already exists for intent ${story.intent.name}")
                    }
                }
            }
            if (storyWithSameNsBotAndName != null && storyWithSameNsBotAndName._id != storyWithSameId?._id
            ) {
                if (storyWithSameNsBotAndName.currentType != builtin) {
                    badRequest("Story ${story.name} (${story.currentType}) already exists")
                }
            }

            val newStory = when {
                storyWithSameId != null -> {
                    mergeStory(storyWithSameId, story, application, botConf.botId)
                }
                storyWithSameNsBotAndIntent != null -> {
                    mergeStory(storyWithSameNsBotAndIntent, story, application, botConf.botId)
                }
                storyWithSameNsBotAndName != null -> {
                    mergeStory(storyWithSameNsBotAndName, story, application, botConf.botId)
                }
                else -> {
                    StoryDefinitionConfiguration(
                        storyId = story.storyId,
                        botId = story.botId,
                        intent = story.intent,
                        currentType = story.currentType,
                        answers = story.answers.mapNotNull { it.toStoryConfiguration(botConf.botId, null) },
                        version = 0,
                        namespace = namespace,
                        mandatoryEntities = story.mandatoryEntities.map {
                            it.toEntityConfiguration(
                                application,
                                botConf.botId,
                                storyWithSameId
                            )
                        },
                        steps = story.steps.map { it.toStepConfiguration(application, botConf.botId, null) },
                        name = story.name,
                        category = story.category,
                        description = story.description,
                        userSentence = story.userSentence,
                        userSentenceLocale = story.userSentenceLocale,
                        configurationName = story.configurationName,
                        features = story.features,
                        tags = story.tags
                    )
                }
            }

            logger.debug { "Saving story: $newStory" }
            storyDefinitionDAO.save(newStory)

            if (story.userSentence.isNotBlank()) {
                val intent = createOrGetIntent(
                    namespace,
                    story.intent.name,
                    application._id,
                    story.category
                )
                saveSentence(story.userSentence, story.userSentenceLocale, application._id, intent._id, user)
            }

            //save all intents of steps
            newStory.steps.forEach { saveUserSentenceOfStep(application, it, user) }

            BotStoryDefinitionConfiguration(newStory, story.userSentenceLocale)
        } else {
            null
        }
    }

    private fun saveSentence(
        text: String,
        locale: Locale,
        applicationId: Id<ApplicationDefinition>,
        intentId: Id<IntentDefinition>,
        user: UserLogin
    ) {

        if (
            front.search(
                SentencesQuery(
                    applicationId = applicationId,
                    language = locale,
                    search = text,
                    onlyExactMatch = true,
                    intentId = intentId,
                    status = setOf(validated, model)
                )
            ).total == 0L
        ) {
            front.save(
                ClassifiedSentence(
                    text = text,
                    language = locale,
                    applicationId = applicationId,
                    creationDate = Instant.now(),
                    updateDate = Instant.now(),
                    status = validated,
                    classification = Classification(intentId, emptyList()),
                    lastIntentProbability = 1.0,
                    lastEntityProbability = 1.0,
                    qualifier = user
                )
            )
        }
    }

    private fun saveUserSentenceOfStep(
        application: ApplicationDefinition,
        step: StoryDefinitionConfigurationStep,
        user: UserLogin
    ) {

        val label = step.userSentenceLabel?.let { Translator.getLabel(it.key) }
        if (label != null && step.intent != null) {
            application.supportedLocales.forEach { locale ->
                val text = label.findLabel(locale)?.label
                    ?: label.findLabel(defaultLocale)?.label
                if (text != null) {
                    val intent = front.getIntentByNamespaceAndName(application.namespace, step.intent!!.name)
                    if (intent != null) {
                        saveSentence(text, locale, application._id, intent._id, user)
                    }
                }
            }
        }

        step.children.forEach { saveUserSentenceOfStep(application, it, user) }
    }

    fun createI18nRequest(namespace: String, request: CreateI18nLabelRequest): I18nLabel {
        val labelKey =
            I18nKeyProvider
                .simpleKeyProvider(namespace, request.category)
                .i18n(request.label)
        return Translator.create(labelKey, request.locale)
    }

    fun getFeatures(botId: String, namespace: String): List<FeatureState> {
        return featureDAO.getFeatures(botId, namespace)
    }

    fun toggleFeature(botId: String, namespace: String, feature: Feature) {
        if (featureDAO.isEnabled(botId, namespace, feature.category, feature.name, feature.applicationId)) {
            featureDAO.disable(botId, namespace, feature.category, feature.name, feature.applicationId)
        } else {
            featureDAO.enable(
                botId,
                namespace,
                feature.category,
                feature.name,
                feature.startDate,
                feature.endDate,
                feature.applicationId
            )
        }
    }

    fun updateDateAndEnableFeature(botId: String, namespace: String, feature: Feature) {
        featureDAO.enable(
            botId,
            namespace,
            feature.category,
            feature.name,
            feature.startDate,
            feature.endDate,
            feature.applicationId
        )
    }

    fun addFeature(botId: String, namespace: String, feature: Feature) {
        featureDAO.addFeature(
            botId = botId,
            namespace = namespace,
            enabled = feature.enabled,
            category = feature.category,
            name = feature.name,
            startDate = feature.startDate,
            endDate = feature.endDate,
            applicationId = feature.applicationId
        )
    }

    fun deleteFeature(botId: String, namespace: String, category: String, name: String, applicationId: String?) {
        featureDAO.deleteFeature(botId, namespace, category, name, applicationId)
    }

    fun loadDialogFlow(request: DialogFlowRequest): ApplicationDialogFlowData {
        val namespace = request.namespace
        val botId = request.botId
        val applicationIds = loadApplicationIds(request)
        logger.debug { "Loading Bot Flow for ${applicationIds.size} configurations: $applicationIds..." }
        return dialogFlowDAO.loadApplicationData(namespace, botId, applicationIds, request.from, request.to)
    }

    private fun loadApplicationIds(request: DialogFlowRequest): Set<Id<BotApplicationConfiguration>> {
        return loadApplications(request).map { it._id }.toSet()
    }

    private fun loadApplications(request: DialogFlowRequest): Set<BotApplicationConfiguration> {
        val namespace = request.namespace
        val botId = request.botId
        val configurationName = request.botConfigurationName
        val tests = request.includeTestConfigurations
        return if (request.botConfigurationId != null) {
            if (tests && configurationName != null) {
                val configurations = applicationConfigurationDAO.getConfigurationsByBotNamespaceAndConfigurationName(
                    namespace = namespace,
                    botId = botId,
                    configurationName = configurationName
                )
                val actualConfiguration = configurations.find { it._id == request.botConfigurationId }
                val testConfiguration =
                    configurations.find { it.applicationId == "test-${actualConfiguration?.applicationId}" }
                listOfNotNull(actualConfiguration, testConfiguration).toSet()
            } else
                listOfNotNull(applicationConfigurationDAO.getConfigurationById(request.botConfigurationId)).toSet()
        } else if (configurationName != null) {
            applicationConfigurationDAO
                .getConfigurationsByBotNamespaceAndConfigurationName(namespace, botId, configurationName)
                .filter { tests || it.connectorType != ConnectorType.rest }
                .toSet()
        } else {
            applicationConfigurationDAO
                .getConfigurationsByNamespaceAndBotId(namespace, botId)
                .filter { tests || it.connectorType != ConnectorType.rest }
                .toSet()
        }
    }

    fun deleteApplication(app: ApplicationDefinition) {
        applicationConfigurationDAO.getConfigurationsByNamespaceAndNlpModel(
            app.namespace, app.name
        ).forEach {
            applicationConfigurationDAO.delete(it)
        }
        applicationConfigurationDAO.getBotConfigurationsByNamespaceAndBotId(
            app.namespace, app.name
        ).forEach {
            applicationConfigurationDAO.delete(it)
        }

        //delete stories
        storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(
            app.namespace, app.name
        ).forEach {
            storyDefinitionDAO.delete(it)
        }
    }

    fun changeApplicationName(existingApp: ApplicationDefinition, newApp: ApplicationDefinition) {
        applicationConfigurationDAO.getConfigurationsByNamespaceAndNlpModel(
            existingApp.namespace, existingApp.name
        ).forEach {
            applicationConfigurationDAO.save(it.copy(botId = newApp.name, nlpModel = newApp.name))
        }
        applicationConfigurationDAO.getBotConfigurationsByNamespaceAndBotId(
            existingApp.namespace, existingApp.name
        ).forEach {
            applicationConfigurationDAO.save(it.copy(botId = newApp.name))
        }
        //stories
        storyDefinitionDAO.getStoryDefinitionsByNamespaceAndBotId(
            existingApp.namespace, existingApp.name
        ).forEach {
            storyDefinitionDAO.save(it.copy(botId = newApp.name))
        }
    }
}
