package ai.vyro.imagine.sdk

import ai.vyro.imagine.sdk.features.AspectRatio
import ai.vyro.imagine.sdk.features.generations.GenerationsHandler
import ai.vyro.imagine.sdk.features.generations.GenerationsStyle
import ai.vyro.imagine.sdk.features.inpaint.InPaintHandler
import ai.vyro.imagine.sdk.features.inpaint.InPaintStyle
import ai.vyro.imagine.sdk.features.remix.ImageRemixControl
import ai.vyro.imagine.sdk.features.remix.ImageRemixStyle
import ai.vyro.imagine.sdk.features.remix.RemixHandler
import ai.vyro.imagine.sdk.features.upscale.UpscaleHandler
import ai.vyro.imagine.sdk.features.upscale.UpscaleStyle
import ai.vyro.imagine.sdk.features.variations.VariationsHandler
import ai.vyro.imagine.sdk.impl.ImagineImpl
import ai.vyro.imagine.sdk.models.Response
import ai.vyro.imagine.sdk.models.response.Image
import ai.vyro.imagine.sdk.plugins.ConfigImpl
import ai.vyro.imagine.sdk.plugins.ImagineConfig
import ai.vyro.imagine.sdk.remote.plugins.Authorization
import io.ktor.client.HttpClient

interface Imagine {
    suspend fun generations(
        prompt: String,
        style: GenerationsStyle = GenerationsStyle.IMAGINE_V3,
        aspectRatio: AspectRatio? = null,
        negPrompt: String? = null,
        cfg: Float? = null,
        seed: Int? = null,
        steps: Int? = null,
        highResolution: Boolean = false
    ): Response<Image>

    suspend fun remix(
        image: ByteArray,
        prompt: String,
        style: ImageRemixStyle = ImageRemixStyle.IMAGINE_V1,
        control: ImageRemixControl = ImageRemixControl.OPEN_POSE,
        negPrompt: String? = null,
        cfg: Float? = null,
        seed: Int? = null,
        steps: Int? = null,
        strength: Int? = null
    ): Response<Image>

    suspend fun upscale(
        image: ByteArray,
        style: UpscaleStyle = UpscaleStyle.BASIC
    ): Response<Image>

    suspend fun variations(
        image: ByteArray,
        prompt: String,
        style: GenerationsStyle = GenerationsStyle.IMAGINE_V3,
        negPrompt: String? = null,
        cfg: Float? = null,
        seed: Int? = null,
        steps: Int? = null,
        strength: Int? = null
    ): Response<Image>

    suspend fun inpaint(
        image: ByteArray,
        mask: ByteArray,
        prompt: String,
        style: InPaintStyle = InPaintStyle.BASIC
    ): Response<Image>
}

@Suppress("FunctionName")
fun ImagineClient(token: String, config: ImagineConfig.() -> Unit = {}): Imagine {
    val client = HttpClient {
        ConfigImpl().apply {
            config()

            ktor {
                install(Authorization(token))
            }
        }.build()(this)
    }

    return ImagineImpl(
        generationsHandler = GenerationsHandler(client),
        remixHandler = RemixHandler(client),
        upscaleHandler = UpscaleHandler(client),
        variationsHandler = VariationsHandler(client),
        inpaintHandler = InPaintHandler(client)
    )
}