package ai.waii;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

public class WaiiHttpClient {

    private final String url;
    private final String apiKey;
    private final long timeout = 150000; // adjusted for milliseconds
    private String scope = "";
    private String orgId = "";
    private String userId = "";
    private final Gson gson = new Gson();


    public WaiiHttpClient(String url, String apiKey) {
        this.url = url != null ? url : "http://localhost:9859/api/";
        this.apiKey = apiKey != null ? apiKey : "";
    }

    public void setScope(String scope) {
        this.scope = scope;
    }

    public String getScope() {
        return this.scope;
    }

    public void setOrgId(String orgId) {
        this.orgId = orgId;
    }

    public String getOrgId() {
        return this.orgId;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public String getUserId() {
        return this.userId;
    }

    public String commonFetchRaw(String endpoint, String jsonParams) throws IOException {
               HttpURLConnection connection = null;
        
        // Convert jsonParams to a Map
        Map<String, Object> paramsMap = gson.fromJson(jsonParams, Map.class);

        // Add scope, orgId, and userId to the params map
        paramsMap.put("scope", this.scope);
        paramsMap.put("org_id", this.orgId);
        paramsMap.put("user_id", this.userId);

        // Convert the modified map back to JSON
        String modifiedJsonParams = gson.toJson(paramsMap);

        try {
            URL apiUrl = new URL(this.url + endpoint);
            connection = (HttpURLConnection) apiUrl.openConnection();
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Authorization", "Bearer " + this.apiKey);
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setConnectTimeout((int) this.timeout);
            connection.setReadTimeout((int) this.timeout);
            connection.setDoOutput(true);

            try (OutputStream os = connection.getOutputStream()) {
                byte[] input = modifiedJsonParams.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }

            int status = connection.getResponseCode();

            BufferedReader reader;
            if (status >= 200 && status < 300) {
                reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            } else {
                reader = new BufferedReader(new InputStreamReader(connection.getErrorStream(), StandardCharsets.UTF_8));
            }

            StringBuilder response = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                response.append(line.trim());
            }

            if (status >= 400 && status < 500) {
               throw new IOException("Authentication failed: Incorrect API key.");
            } else if (status >= 500) {
                throw new IOException("Server error: " + response.toString());
            }

            return response.toString();
        } catch (JsonSyntaxException | IOException e) {
            throw new IOException("Invalid response received.", e);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    public <Type> Type commonFetch(String endpoint, String jsonParams, Class<Type> clazz) throws IOException {
        String response = commonFetchRaw(endpoint, jsonParams);
        //System.out.println(response);
        return gson.fromJson(response, clazz);
    }
}
