package com.wordbox.ai.common.network.di

import android.util.Log
import com.chuckerteam.chucker.api.ChuckerInterceptor
import com.wordbox.ai.common.network.FirebaseUserIdTokenInterceptor
import com.wordbox.network.constants.NetworkConstants.CHUCKER_MAX_CONTENT_LENGTH
import com.wordbox.network.constants.NetworkConstants.TIME_OUT
import io.ktor.client.HttpClient
import io.ktor.client.engine.okhttp.OkHttp
import io.ktor.client.features.DefaultRequest
import io.ktor.client.features.json.JsonFeature
import io.ktor.client.features.json.serializer.KotlinxSerializer
import io.ktor.client.features.logging.LogLevel
import io.ktor.client.features.logging.Logger
import io.ktor.client.features.logging.Logging
import io.ktor.client.features.observer.ResponseObserver
import io.ktor.client.request.header
import io.ktor.http.ContentType
import io.ktor.http.HttpHeaders
import kotlinx.serialization.json.Json
import org.koin.core.module.Module
import org.koin.dsl.module
import java.util.concurrent.TimeUnit.SECONDS

val networkModule: Module = module {

    single {
        HttpClient(OkHttp) {
            install(DefaultRequest) {
                header(HttpHeaders.ContentType, ContentType.Application.Json)
            }
            install(JsonFeature) {
                serializer = KotlinxSerializer(
                    Json {
                        prettyPrint = true
                        isLenient = true
                        ignoreUnknownKeys = true
                    }
                )
            }
            engine {
                addInterceptor(FirebaseUserIdTokenInterceptor())
                addInterceptor(
                    ChuckerInterceptor(
                        context = get(),
                        maxContentLength = CHUCKER_MAX_CONTENT_LENGTH
                    )
                )
                config {
                    connectTimeout(TIME_OUT, SECONDS)
                    readTimeout(TIME_OUT, SECONDS).build()
                }
            }
            install(ResponseObserver) {
                onResponse { response ->
                    Log.d("HTTP status:", "${response.status.value}")
                }
            }
            install(Logging) {
                logger = object : Logger {
                    override fun log(message: String) {
                        Log.v("Logger Ktor =>", message)
                    }
                }
                level = LogLevel.ALL
            }
        }
    }
}
