package ai.yue.library.data.jdbc.support;

import ai.yue.library.base.util.MapUtils;
import ai.yue.library.data.jdbc.client.DbBase;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.fastjson.JSONObject;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.LinkedCaseInsensitiveMap;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;

/**
 * 参考 {@linkplain org.springframework.jdbc.core.ColumnMapRowMapper}，替换为 fastjson 的 {@linkplain JSONObject}
 * 
 * @author	ylyue
 * @since	2020年11月2日
 */
public class ColumnMapRowMapper implements RowMapper<JSONObject> {

	private DbBase dbBase;
	@Nullable
	private String[] tableNames;

	public ColumnMapRowMapper(DbBase dbBase, String... tableNames) {
		this.dbBase = dbBase;
		this.tableNames = tableNames;
	}

	@Override
	public JSONObject mapRow(ResultSet rs, int rowNum) throws SQLException {
		ResultSetMetaData rsmd = rs.getMetaData();
		int columnCount = rsmd.getColumnCount();
		JSONObject resultJson = createColumnMap(columnCount);
		for (int i = 1; i <= columnCount; i++) {
			String column = JdbcUtils.lookupColumnName(rsmd, i);
			resultJson.putIfAbsent(getColumnKey(column), getColumnValue(rs, i));
		}

		if (ArrayUtil.isNotEmpty(tableNames) && MapUtils.isNotEmpty(resultJson)) {
			dbBase.aopAfter(tableNames, resultJson);
		}

		return resultJson;
	}

	/**
	 * Create a Map instance to be used as column map.
	 * <p>By default, a linked case-insensitive Map will be created.
	 * @param columnCount the column count, to be used as initial
	 * capacity for the Map
	 * @return the new Map instance
	 * @see org.springframework.util.LinkedCaseInsensitiveMap
	 */
	protected JSONObject createColumnMap(int columnCount) {
		return new JSONObject(new LinkedCaseInsensitiveMap<>(columnCount));
	}

	/**
	 * Determine the key to use for the given column in the column Map.
	 * @param columnName the column name as returned by the ResultSet
	 * @return the column key to use
	 * @see java.sql.ResultSetMetaData#getColumnName
	 */
	protected String getColumnKey(String columnName) {
		return columnName;
	}

	/**
	 * Retrieve a JDBC object value for the specified column.
	 * <p>The default implementation uses the {@code getObject} method.
	 * Additionally, this implementation includes a "hack" to get around Oracle
	 * returning a non standard object for their TIMESTAMP datatype.
	 * @param rs is the ResultSet holding the data
	 * @param index is the column index
	 * @return the Object returned
	 * @see org.springframework.jdbc.support.JdbcUtils#getResultSetValue
	 */
	@Nullable
	protected Object getColumnValue(ResultSet rs, int index) throws SQLException {
		return JdbcUtils.getResultSetValue(rs, index);
	}

}
