package com.alan.alansdk.alanbase.speaker;

import android.annotation.SuppressLint;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioTrack;
import android.os.Process;

import com.alan.alansdk.Utils;
import com.alan.alansdk.alanbase.AudioConfig;
import com.alan.alansdk.logging.AlanLogger;

import okhttp3.internal.Util;

public class SpeakerThread extends Thread {

    private AudioTrack audioTrack;

    private boolean shouldSpeak = false;
    private final SpeakerDumpListener callback;
    private boolean shouldStop = false;

    public SpeakerThread(SpeakerDumpListener callback) {
        this.callback = callback;
        android.os.Process.setThreadPriority(Process.THREAD_PRIORITY_URGENT_AUDIO);
    }

    public void cancel() {
        shouldStop = true;
    }

    @Override
    public void run() {
        super.run();
        while (!shouldStop) {
            try {
                sleep(200);
            } catch (InterruptedException e) {
                readFrames();
            }
        }
    }

    public void release() {
        shouldSpeak = false;
        shouldStop = true;
        if (audioTrack != null) {
            audioTrack.flush();
            audioTrack.release();
            audioTrack = null;
        }
    }

    public void stopPlaying() {
        shouldSpeak = false;
    }

    @SuppressLint("NewApi")
    public void readFrames() {
        shouldSpeak = true;
        if (audioTrack == null || audioTrack.getState() == AudioTrack.STATE_UNINITIALIZED) {
            AlanLogger.w("Reading frames from null or not initialized audiotrack");
            if (audioTrack != null) {
                audioTrack.release();
            }
            preparePlayer();
            try {
                sleep(200);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            return;
        }
        audioTrack.play();
        AlanLogger.d("Audio streaming started");

        float[] buffer = callback.retrieveAudioBuffer();
        while (shouldSpeak) {
            if (buffer != null) {
                if (AudioConfig.AUDIO_FORMAT == AudioFormat.ENCODING_PCM_FLOAT) {
                    audioTrack.write(buffer, 0, buffer.length, AudioTrack.WRITE_BLOCKING);
                } else {
                    short[] shorts = Utils.floatsToShorts(buffer);
                    audioTrack.write(shorts, 0, shorts.length, AudioTrack.WRITE_BLOCKING);
                }
            }
            buffer = callback.retrieveAudioBuffer();
        }
    }

    private boolean preparePlayer() {
        int bufferSize = AudioTrack.getMinBufferSize(
                AudioConfig.SAMPLE_RATE,
                AudioFormat.CHANNEL_OUT_MONO,
                AudioConfig.AUDIO_FORMAT
        );
        if (bufferSize == AudioTrack.ERROR || bufferSize == AudioTrack.ERROR_BAD_VALUE) {
            bufferSize = AudioConfig.SAMPLE_RATE * 2;
        }
        AlanLogger.i("Speaker thread - use buffer size - " + bufferSize);

        audioTrack = new AudioTrack(
                AudioManager.STREAM_MUSIC,
                AudioConfig.SAMPLE_RATE,
                AudioFormat.CHANNEL_OUT_MONO,
                AudioConfig.AUDIO_FORMAT,
                bufferSize,
                AudioTrack.MODE_STREAM);
        try {
            sleep(100);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return true;
    }
}
