package com.alan.alansdk.screenshot;

import android.content.Context;

import com.alan.alansdk.logging.AlanLogger;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class UploadThread extends Thread {

    private final Context context;
    private final String server;
    private File rootDir;

    private final OkHttpClient client = new OkHttpClient();
    private final MediaType MEDIA_TYPE_MARKDOWN
            = MediaType.parse("text/plain;");

    public UploadThread(
            Context context,
            String server) {
        this.server = server.replaceFirst("wss", "https");
        this.context = context;
    }

    @Override
    public void run() {
        super.run();
        AlanLogger.i("Starting pending screenshot upload");
        rootDir = new File(context.getFilesDir(), ScreenShooter.SCREENSHOTS_FOLDER);

        List<String> screenshots = collectScreenshots();
        for (String screenshotName : screenshots) {
            uploadScreenshot(screenshotName);
        }
    }

    private List<String> collectScreenshots() {
        if (rootDir == null) {
            AlanLogger.e("Cannot open screenshots folder");
            return new LinkedList<>();
        } else {
            String[] fileList = rootDir.list();
            if (fileList == null) {
                AlanLogger.i("Screenshot folder is empty");
                return new LinkedList<>();
            } else {
                AlanLogger.i("Got " + fileList.length + " pending screenshots");
                return Arrays.asList(fileList);
            }
        }
    }

    private File getScreenshotByName(String name) {
        File screenshot = new File(rootDir, name);
        while (screenshot.isDirectory() && screenshot.list().length > 0) {
            screenshot = new File(screenshot, screenshot.list()[0]);
        }
        return screenshot;
    }

    private void uploadScreenshot(String name) {
        OutputStream out = null;
        File screenshot = getScreenshotByName(name);
        String screenshotPath = screenshot.getAbsolutePath();
        if (screenshotPath == null) {
            AlanLogger.i("Screenshot path is empty");
            return;
        }
        int mediaAssetIndex = screenshotPath.indexOf("media-asset/");
        if (mediaAssetIndex < 0) {
            mediaAssetIndex = screenshotPath.indexOf("media-asset-stage/");
        }
        if (mediaAssetIndex < 0) {
            AlanLogger.i("Can't find valid filename in the path");
            return;
        }
        String screenshotName = screenshot.getAbsolutePath().substring(mediaAssetIndex);
        String urlString = server + "/upload-media/" + screenshotName;
        AlanLogger.i("Uploading to: " + server + '/' + screenshotName);

        boolean result = upload(screenshot, urlString);
        if (result) {
            AlanLogger.i("Uploaded successfully");
            deleteDirectory(name);
        } else {
            AlanLogger.i("Upload failed");
        }
    }

    private void deleteDirectory(String name) {
        File dir = new File(rootDir, name);
        deleteRecursive(dir);
    }

    private void deleteRecursive(File fileOrDirectory) {
        if (fileOrDirectory.isDirectory()) {
            File[] listOfFlies = fileOrDirectory.listFiles();
            if (listOfFlies != null) {
                for (File child : listOfFlies) {
                    deleteRecursive(child);
                }
            }
            else {
                AlanLogger.i("Upload failed: listFiles() returns null");
            }
        }
        fileOrDirectory.delete();
    }

    private boolean upload(File file, String url) {
        Request request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(MEDIA_TYPE_MARKDOWN, file))
                .build();

        try (Response response = client.newCall(request).execute()) {
            if (!response.isSuccessful()) throw new IOException("Unexpected code " + response);
        } catch (IOException e) {
            AlanLogger.e(e);
        }
        return true;
    }
}
