package app.appnomix.sdk.external

import android.accessibilityservice.AccessibilityServiceInfo
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.provider.Settings
import android.view.accessibility.AccessibilityManager
import androidx.annotation.DrawableRes
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.WorkManager
import app.appnomix.sdk.internal.CouponsAccessibilityService
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.network.CouponsSyncWorker
import app.appnomix.sdk.internal.data.network.LocationSyncWorker
import app.appnomix.sdk.internal.domain.CouponsUpdateListener
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.domain.model.toExternalCoupon
import app.appnomix.sdk.internal.ui.onboarding.OnboardingActivity
import java.lang.ref.WeakReference
import java.util.concurrent.TimeUnit

object CouponsSdkFacade : CouponsUpdateListener {
    private val couponsRepo by lazy { Deps.repo() }
    private val sdkConfig by lazy { Deps.sdkConfig() }
    private var weakListener: WeakReference<ExternalCouponsReadyListener?>? = null
    private lateinit var appContext: Context

    fun initialize(context: Context) {
        appContext = context
        schedulePeriodicCouponsSync(context)
        schedulePeriodicLocationRefresh(context)
    }

    fun setup(config: Config) {
        sdkConfig.appIconRes = config.appIconRes
        sdkConfig.authToken = config.authToken
        sdkConfig.clientId = config.clientId
    }

    private fun schedulePeriodicCouponsSync(context: Context) {
        val workManager = WorkManager.getInstance(context)
        workManager.enqueueUniquePeriodicWork(
            CouponsSyncWorker.NAME,
            ExistingPeriodicWorkPolicy.KEEP,
            CouponsSyncWorker.create(TimeUnit.MILLISECONDS.toHours(sdkConfig.dataRefreshPeriodMillis))
        )
    }

    private fun schedulePeriodicLocationRefresh(context: Context) {
        val workManager = WorkManager.getInstance(context)
        workManager.enqueueUniquePeriodicWork(
            LocationSyncWorker.NAME,
            ExistingPeriodicWorkPolicy.KEEP,
            LocationSyncWorker.create()
        )
    }

    fun unmetRequirements(): List<SdkRequirement> {
        val isAccessibilityServiceEnabled = isAccessibilityServiceEnabled()

        return mutableListOf<SdkRequirement>().apply {
            if (!isAccessibilityServiceEnabled)
                add(SdkRequirement.ACCESSIBILITY)
        }
    }

    fun isAccessibilityServiceEnabled(
    ): Boolean {
        val accessibilityManager =
            appContext.getSystemService(Context.ACCESSIBILITY_SERVICE) as AccessibilityManager

        val enabledServices = accessibilityManager.getEnabledAccessibilityServiceList(
            AccessibilityServiceInfo.FEEDBACK_ALL_MASK
        )

        for (enabledService in enabledServices) {
            val serviceInfo = enabledService.resolveInfo.serviceInfo
            if (serviceInfo.packageName == appContext.packageName && serviceInfo.name == CouponsAccessibilityService::class.java.name) {
                return true
            }
        }

        return false
    }

    fun registerCouponsReadyListener(listener: ExternalCouponsReadyListener) {
        weakListener = WeakReference(listener)
        couponsRepo.setListener(this)
    }

    override fun onCouponsUpdated(coupons: List<Coupon>) {
        weakListener?.get()?.onCouponsAvailable(coupons
            .filter { !it.isForHomepage }
            .map { it.toExternalCoupon() })
    }

    fun launchSdkOnboardingActivity(activity: Activity) {
        if (!sdkConfig.onboardingFinished) {
            val intent = Intent(activity, OnboardingActivity::class.java)
            activity.startActivity(intent)
        }
    }

    fun goToToAccessibilityOnboarding(activity: Activity) {
        val intent = Intent(activity, OnboardingActivity::class.java)
        intent.action = OnboardingActivity.TO_ACCESSIBILITY_STEP_ACTION
        activity.startActivity(intent)
    }

    fun goToToAccessibilitySettings(activity: Activity) {
        val intent = Intent(Settings.ACTION_ACCESSIBILITY_SETTINGS)
        activity.startActivity(intent)
    }

    data class Config(
        @DrawableRes val appIconRes: Int = 0,
        val authToken: String,
        val clientId: String
    )
}

interface ExternalCouponsReadyListener {
    fun onCouponsAvailable(coupons: List<ExternalCoupon>)
}
