package app.appnomix.sdk.internal.domain

import android.content.Context
import androidx.room.withTransaction
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.local.SaversLeagueDatabase
import app.appnomix.sdk.internal.data.local.dao.CouponDao
import app.appnomix.sdk.internal.data.local.dao.DemandDao
import app.appnomix.sdk.internal.data.local.model.toCoupon
import app.appnomix.sdk.internal.data.local.model.toDemand
import app.appnomix.sdk.internal.data.network.DataResponse
import app.appnomix.sdk.internal.data.network.SaversLeagueApi
import app.appnomix.sdk.internal.data.network.model.CouponDto
import app.appnomix.sdk.internal.data.network.model.DemandDto
import app.appnomix.sdk.internal.data.network.model.toEntity
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.domain.model.Demand
import app.appnomix.sdk.internal.utils.SLog
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.lang.ref.WeakReference
import java.time.LocalDateTime
import java.time.ZoneOffset
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.Executors

class CouponsRepo(
    private val context: Context,
    private val sdkConfig: SdkConfig,
    private val api: SaversLeagueApi
) {
    private val dispatcher = Executors.newSingleThreadExecutor().asCoroutineDispatcher()
    private val scope = CoroutineScope(dispatcher)

    private val db: SaversLeagueDatabase by lazy { SaversLeagueDatabase.instance(context) }
    private val couponDao: CouponDao by lazy { db.couponsDao() }
    private val demandDao: DemandDao by lazy { db.demandsDao() }

    private var couponsCache = CopyOnWriteArrayList<Coupon>()
    private var demandsCache = CopyOnWriteArrayList<Demand>()

    private var weakListener: WeakReference<CouponsUpdateListener?>? = null
    private val exceptionHandler = CoroutineExceptionHandler { _, exception ->
        SLog.e("Something went wrong while doing repo ops", exception)
    }

    init {
        populateCache()
    }

    suspend fun fetchCoupons(countryCode: String?, forHomepage: Boolean): Unit =
        withContext(dispatcher) {
            try {
                val dataResponse: DataResponse<CouponDto> = api.getCoupons(countryCode, forHomepage)
                val currentCountryCode = sdkConfig.countryCode
                val entities = dataResponse.data
                    .filter { LocalDateTime.now() in it.startDate..it.endDate }
                    .filter { coupon ->
                        currentCountryCode.isNullOrEmpty() || coupon.validCountriesCodes
                            .contains(currentCountryCode)
                    }
                    .map { it.toEntity(forHomepage) }
                db.withTransaction {
                    val previousItems = couponDao.getAll()
                    val updatedNewItems = entities.map { newItem ->
                        val previousItem = previousItems.find { it.id == newItem.id }
                        newItem.copy(
                            snoozeTime = previousItem?.snoozeTime,
                            isHomepageCoupon = previousItem?.isHomepageCoupon ?: false
                        )
                    }
                    couponDao.replace(updatedNewItems)

                    val coupons =
                        updatedNewItems.map { it.toCoupon(api.getCouponImageUrl(it.brandDomain)) }
                    couponsCache.clear()
                    couponsCache.addAll(coupons)
                    weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
                }
            } catch (t: Throwable) {
                SLog.e("Something went wrong while saving coupons", t)
            }
        }

    suspend fun fetchDemands(countryCode: String?): Unit = withContext(dispatcher) {
        try {
            val dataResponse: DataResponse<DemandDto> = api.getOnDemandRedirects(countryCode)
            val currentCountryCode = sdkConfig.countryCode
            val entities = dataResponse.data
                .filter { demand ->
                    currentCountryCode.isNullOrEmpty()
                            || demand.countryCode.isNullOrEmpty()
                            || demand.countryCode.lowercase() == currentCountryCode
                }
                .map { it.toEntity() }
            db.withTransaction {
                val previousItems = demandDao.getAll()
                val updatedNewItems = entities.map { newItem ->
                    val previousItem = previousItems.find { it.id == newItem.id }
                    newItem.copy(
                        snoozeTime = previousItem?.snoozeTime,
                        snoozeCount = previousItem?.snoozeCount ?: 0,
                    )
                }
                demandDao.replace(updatedNewItems)

                val demands =
                    updatedNewItems.map { it.toDemand(api.getCouponImageUrl(it.rootDomain)) }
                demandsCache.clear()
                demandsCache.addAll(demands)
            }
        } catch (t: Throwable) {
            SLog.e("Something went wrong while saving demands", t)
        }
    }

    fun matchByUrlAndLocation(
        url: String,
        countryCode: String?
    ): List<Any> {
        val matchedCoupons = couponsCache.filter {
            it.matchesLocation(countryCode) &&
                    it.isValid() &&
                    it.matchesTarget(url) &&
                    !isBrandSnoozed(it)
        }

        return matchedCoupons.ifEmpty {
            demandsCache.filter {
                it.matchesLocation(countryCode) &&
                        it.matchesTarget(url) &&
                        !isBrandSnoozed(it)
            }
        }
    }

    fun markAsSnoozed(coupon: Coupon) {
        scope.launch(exceptionHandler) {
            val couponBrands = couponsCache.filter { it.brandDomain == coupon.brandDomain }.toSet()
            couponsCache.removeAll(couponBrands)

            val snoozedCoupons = couponBrands.map { it.copy(snoozeTime = LocalDateTime.now()) }
            couponsCache.addAll(snoozedCoupons)

            couponDao.updateBrandSnooze(
                time = LocalDateTime.now(),
                brandDomain = coupon.brandDomain
            )
            weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
        }
    }

    fun markAsSnoozed(demand: Demand) {
        scope.launch(exceptionHandler) {
            val couponBrands = demandsCache.filter { it.rootDomain == demand.rootDomain }.toSet()
            demandsCache.removeAll(couponBrands)

            val snoozedCoupons = couponBrands.map { it.copy(snoozeTime = LocalDateTime.now()) }
            demandsCache.addAll(snoozedCoupons)

            demandDao.updateBrandSnooze(
                time = LocalDateTime.now(),
                domain = demand.rootDomain,
                snoozeCount = demand.snoozeCount + 1
            )
        }
    }

    private fun isBrandSnoozed(item: Any): Boolean {
        return when (item) {
            is Coupon -> {
                demandsCache
                    .filter { it.rootDomain == item.brandDomain }
                    .filter { it.snoozeTime != null }
                    .maxByOrNull { it.snoozeTime?.toEpochSecond(ZoneOffset.UTC) ?: 0 }
                    ?.isSnoozed() == true || item.isSnoozed()
            }

            is Demand -> {
                couponsCache
                    .filter { it.brandDomain == item.rootDomain }
                    .filter { it.snoozeTime != null }
                    .maxByOrNull { it.snoozeTime?.toEpochSecond(ZoneOffset.UTC) ?: 0 }
                    ?.isSnoozed() == true || item.isSnoozed()
            }

            else -> {
                false
            }
        }
    }

    private fun populateCache() {
        scope.launch(exceptionHandler) {
            populateCouponsCache()
            populateDemandsCache()
        }
    }

    private suspend fun populateCouponsCache() {
        val coupons = couponDao.getAll()
        couponsCache.clear()
        couponsCache.addAll(coupons.map { it.toCoupon(api.getCouponImageUrl(it.brandDomain)) })
        weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
    }

    private suspend fun populateDemandsCache() {
        val coupons = demandDao.getAll()
        demandsCache.clear()
        demandsCache.addAll(coupons.map { it.toDemand(api.getCouponImageUrl(it.rootDomain)) })
    }

    fun getById(id: String): Coupon? {
        return couponsCache.find { it.id == id }
    }

    fun setListener(listener: CouponsUpdateListener) {
        weakListener = WeakReference(listener)
        if (couponsCache.isNotEmpty()) {
            weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
        }
    }
}

interface CouponsUpdateListener {
    fun onCouponsUpdated(coupons: List<Coupon>)
}
