package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.PixelFormat
import android.graphics.Rect
import android.os.Build
import android.view.LayoutInflater
import android.view.View
import android.view.WindowInsets
import android.view.WindowManager
import android.webkit.WebView
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.utils.SLog

const val DEBUG_WEBVIEW_ENABLED = false

internal class WebviewWindow(private val context: Context) {
    private val windowManager: WindowManager by lazy {
        context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    }
    private var webView: WebView? = null
    private var lastBounds: Rect? = null
    private var lastUrl: String? = null
    private var windowInsets: WindowInsets? = null

    fun update(
        url: String,
        bounds: Rect?
    ) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        if (bounds == lastBounds && url == lastUrl) return
        if (bounds == null) {
            if (webView?.parent != null)
                windowManager.removeView(webView)
            return
        }
        if (webView == null) {
            webView = LayoutInflater.from(context).inflate(R.layout.webview, null) as? WebView
            webView?.settings?.javaScriptEnabled = true
        }

        val topOffset = windowInsets?.topInset ?: 0
        val bottomOffset = windowInsets?.bottomInset ?: 0

        val params = WindowManager.LayoutParams(
            bounds.width(),
            bounds.height(),
            bounds.left,
            bounds.top - topOffset - bottomOffset,
            WindowManager.LayoutParams.TYPE_ACCESSIBILITY_OVERLAY,
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
                    or WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE,
            PixelFormat.TRANSLUCENT
        )

        if (webView?.parent != null) {
            windowManager.removeView(webView)
        }

        webView?.setOnApplyWindowInsetsListener { view, insets ->
            if (windowInsets == null || (insets.topInset > 0 || insets.bottomInset > 0))
                windowInsets = insets
            insets
        }
        windowManager.addView(webView, params)

        webView?.loadUrl(if (url.startsWith("http")) url else "https://$url")
        lastUrl = url
        lastBounds = bounds
    }

    fun click(
        viewIdResourceName: String?,
        contentDescription: CharSequence?,
        text: CharSequence?,
        className: CharSequence?,
    ) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        if (webView != null) {
            val javascript = """
      (function(viewIdResourceName, contentDescription, text, className) {
            var element = null;

            if (viewIdResourceName) {
                element = document.getElementById(viewIdResourceName);
            }

            if (!element && contentDescription) {
                element = document.querySelector('[aria-label="' + contentDescription + '"]');
            }

            if (!element && text) {
                var xpathResult = document.evaluate(
                    '//*[text()="' + text + '"]',
                    document,
                    null,
                    XPathResult.FIRST_ORDERED_NODE_TYPE,
                    null
                );
                element = xpathResult.singleNodeValue;
            }

            if (!element && className) {
                var elements = document.getElementsByClassName(className);
                if (elements.length === 1) {
                    element = elements[0];
                } else if (elements.length > 1) {
                    // Further refine the search if multiple elements have the same className
                    for (var i = 0; i < elements.length; i++) {
                        var el = elements[i];
                        if (contentDescription && el.getAttribute('aria-label') === contentDescription) {
                            element = el;
                            break;
                        } else if (text && el.textContent.trim() === text) {
                            element = el;
                            break;
                        }
                    }
                }
            }

            if (element) {
                element.click();
                return "Element clicked.";
            } else {
                return "Element not found.";
            }
        })(
            ${viewIdResourceName?.let { "\"$it\"" } ?: "null"},
            ${contentDescription?.let { "\"$it\"" } ?: "null"},
            ${text?.let { "\"$it\"" } ?: "null"},
            ${className?.let { "\"$it\"" } ?: "null"}
        );
    """
            webView?.evaluateJavascript(javascript, null)
        }
    }

    fun updateScroll(x: Int, y: Int) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        SLog.i("scroll to: x=$x y=$y")
        webView?.scrollTo(x, y)
    }

    fun hide() {
        webView?.visibility = View.GONE
    }
}

@Suppress("DEPRECATION")
val WindowInsets.topInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).top
    } else {
        this.systemWindowInsetTop
    }

@Suppress("DEPRECATION")
val WindowInsets.bottomInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).bottom
    } else {
        this.systemWindowInsetBottom
    }
