package app.appnomix.sdk.internal.domain.model

import java.time.LocalDateTime
import kotlin.time.Duration.Companion.days
import kotlin.time.Duration.Companion.hours

private const val FALLBACK_TRIGGER_SNOOZE_COUNT = 3
private val DEFAULT_SNOOZE_INTERVAL = 14.days
private val FALLBACK_SNOOZE_INTERVAL = 90.days

data class Demand(
    val redirectUrl: String,
    val rootDomain: String,
    val countryCode: String?,
    val brandImageUrl: String?,
    val snoozeTime: LocalDateTime?,
    val snoozeCount: Int,
) {
    private val regex: Regex by lazy {
        Regex(
            pattern = "^(?:https?://(?:www\\.)?)?(.*\\.)*$rootDomain.*",
            options = setOf(RegexOption.IGNORE_CASE)
        )
    }

    fun matchesTarget(address: String): Boolean {
        return regex.matches(address)
    }

    fun matchesLocation(countryCode: String?): Boolean {
        if (countryCode == null) {
            return true
        }
        return countryCode.lowercase() == countryCode
    }

    fun isSnoozed(): Boolean {
        if (snoozeTime == null) return false
        val intervalToUse = if (snoozeCount >= FALLBACK_TRIGGER_SNOOZE_COUNT)
            FALLBACK_SNOOZE_INTERVAL.inWholeMilliseconds else DEFAULT_SNOOZE_INTERVAL.inWholeMilliseconds

        return LocalDateTime.now()
            .isBefore(snoozeTime.plusSeconds(intervalToUse))
    }
}