package app.appnomix.sdk.internal.domain.browsers

import android.content.Context
import android.content.Intent
import android.graphics.Rect
import android.net.Uri
import android.os.Build
import android.view.accessibility.AccessibilityNodeInfo
import android.webkit.WebView
import android.widget.EditText
import androidx.core.os.bundleOf
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.ui.DEBUG_WEBVIEW_ENABLED
import app.appnomix.sdk.internal.utils.SLog

internal abstract class DeviceBrowser {
    private var urlCouponWasAppliedOn: String? = null

    abstract fun browserPackage(): String
    abstract fun openUrl(
        context: Context,
        currentUrl: String?,
        destinationUrl: String,
        sdkConfig: SdkConfig
    )

    open fun getAddressText(nodeInfo: AccessibilityNodeInfo): String? = null
    open fun isIdle(nodeInfo: AccessibilityNodeInfo): Boolean = false

    fun findInfos(nodeInfo: AccessibilityNodeInfo, tag: String): List<AccessibilityNodeInfo> {
        return nodeInfo.findAccessibilityNodeInfosByViewId(tag)
    }

    fun findByClass(nodeList: List<AccessibilityNodeInfo>, className: String): Boolean {
        for (node in nodeList) {
            if (node.className == className) {
                return true
            }
        }
        return false
    }

    fun isOnCheckoutPage(addressUrl: String) = CHECKOUT_PAGE_REGEX.matches(addressUrl)

    fun applyCouponCodeOnce(
        node: AccessibilityNodeInfo?,
        url: String,
        coupon: String,
    ): Boolean {
        try {
            if (url != urlCouponWasAppliedOn) {
                return applyCouponCodeOnField(node, coupon, url)
            }
        } catch (t: Throwable) {
            SLog.e("Could not apply coupon", t)
        }

        return false
    }

    private fun applyCouponCodeOnField(
        node: AccessibilityNodeInfo?, coupon: String, url: String
    ): Boolean {
        if (node == null) return false

        if (node.className == EditText::class.java.name && node.isEditable) {
            val idName = node.viewIdResourceName?.lowercase() ?: ""
            val hint = kotlin.runCatching {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
                    node.hintText?.toString()?.lowercase() else ""
            }.getOrNull() ?: ""

            SLog.d("editable field: $idName / \"$hint\"")

            val matchesField = COUPON_CODE_INPUT_FIELD_MATCHERS.any {
                idName.contains(
                    it,
                    ignoreCase = true
                ) || hint.contains(it, ignoreCase = true)
            }

            if (matchesField) {
                node.performAction(
                    AccessibilityNodeInfo.ACTION_SET_TEXT,
                    bundleOf(AccessibilityNodeInfo.ACTION_ARGUMENT_SET_TEXT_CHARSEQUENCE to coupon)
                )
                node.performAction(AccessibilityNodeInfo.ACTION_CLICK)
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    node.performAction(AccessibilityNodeInfo.AccessibilityAction.ACTION_IME_ENTER.id)
                }

                urlCouponWasAppliedOn = url
                SLog.d("injected $coupon on $urlCouponWasAppliedOn")
                return true
            }
        }

        if (node.childCount > 0) {
            for (i in 0 until node.childCount) {
                val applied = applyCouponCodeOnField(node.getChild(i), coupon, url)
                if (applied) {
                    return true
                }
            }
        }

        return false
    }

    fun locateWebviewBounds(node: AccessibilityNodeInfo?): Rect? {
        if (node == null || !DEBUG_WEBVIEW_ENABLED) {
            return null
        }

        if (node.className == WebView::class.java.name) {
            return Rect().apply {
                node.getBoundsInScreen(this)
            }
        }

        for (i in 0 until node.childCount) {
            val childNode = node.getChild(i)
            val result = locateWebviewBounds(childNode)
            if (result != null) {
                return result
            }
        }

        return null
    }

    fun replaceRedirectUrlFromFullUrl(
        fullUrl: String,
        currentUrl: String,
        sdkConfig: SdkConfig
    ): String {
        try {
            val uri = Uri.parse(fullUrl)
            val redirectUrl = uri.getQueryParameter("url")
            if (!redirectUrl.isNullOrEmpty()) {
                val queryToReuse = uri.queryParameterNames
                    .filter { it != "url" }
                    .joinToString("&") {
                        "$it=${uri.getQueryParameter(it)}"
                    }

                val newUri = uri.buildUpon()
                    .clearQuery()
                    .appendQueryParameter("url", currentUrl.ensureHttpsPrefix())
                    .build()
                    .toString()
                    .removeSuffix("/")
                    .plus("&$queryToReuse")
                    .plus(
                        if (sdkConfig.adId.isNotEmpty())
                            "&es=${sdkConfig.adId}"
                        else if (sdkConfig.installationId.isNotEmpty())
                            "&ec=${sdkConfig.installationId}"
                        else ""
                    )
                return newUri
            }

        } catch (e: Exception) {
            SLog.e("Unable to replace redirect url from: $fullUrl", e)
        }
        return fullUrl
    }

    companion object {
        private val CHECKOUT_PAGE_REGEX = Regex(
            pattern = "(?:https?://(?:www\\.)?)?(.*\\.)*.*\\/(?:.*checkout.*|.*cart.*|.*basket.*)",
            options = setOf(RegexOption.IGNORE_CASE)
        )

        private val COUPON_CODE_INPUT_FIELD_MATCHERS = listOf(
            "discount", "coupon", "code", "voucher"
        )

        fun openUrl(context: Context, url: String) {
            try {
                val intent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                context.startActivity(intent)
            } catch (e: Exception) {
                SLog.e("Unable to open url $url", e)
            }
        }
    }
}

fun String.ensureHttpsPrefix(): String {
    return this
        .replace(Regex("^http://"), "https://")
        .replace(Regex("^(?!https://www\\.)"), "https://www.")
}