package app.appnomix.sdk.internal.analytics

import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.ExistingWorkPolicy
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.network.AnalyticsSendWorker
import app.appnomix.sdk.internal.data.network.AnalyticsSendWorker.Companion.ANALYTICS_TAG
import app.appnomix.sdk.internal.utils.SLog
import java.util.concurrent.CopyOnWriteArrayList

object Analytics {
    private val cachedEvents = CopyOnWriteArrayList<AnalyticsEvent>()

    fun trackEvent(event: AnalyticsEvent) {
        SLog.i("$ANALYTICS_TAG Tracking ${event.category}/${event.name}")
        cachedEvents += event
        schedulePeriodicSend()

        if (cachedEvents.size >= 20) {
            scheduleOneTimeSend()
        }
    }

    fun getTrackedEvents(): List<AnalyticsEvent> = cachedEvents.toList()

    fun clearTrackedEvents(events: List<AnalyticsEvent>) {
        cachedEvents.removeAll(events.toSet())
    }

    private fun schedulePeriodicSend() {
        Deps.workManager()?.enqueueUniquePeriodicWork(
            /* uniqueWorkName = */ AnalyticsSendWorker.NAME,
            /* existingPeriodicWorkPolicy = */ ExistingPeriodicWorkPolicy.KEEP,
            /* periodicWork = */ AnalyticsSendWorker.createPeriodicWork(),
        )
    }

    private fun scheduleOneTimeSend() {
        Deps.workManager()?.enqueueUniqueWork(
            /* uniqueWorkName = */ "${AnalyticsSendWorker.NAME}-${cachedEvents.size}",
            /* existingWorkPolicy = */ ExistingWorkPolicy.APPEND,
            /* work = */ AnalyticsSendWorker.createUniqueWorkRequest(),
        )
    }
}
