package app.appnomix.sdk.internal.data.network

import android.content.Context
import android.content.pm.PackageManager
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequest
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.PeriodicWorkRequest
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkerParameters
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.analytics.Analytics
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.network.model.AnalyticsEventsDto
import app.appnomix.sdk.internal.data.network.model.toDto
import app.appnomix.sdk.internal.utils.SLog
import io.ktor.http.HttpStatusCode
import java.util.concurrent.TimeUnit

class AnalyticsSendWorker(private val appContext: Context, params: WorkerParameters) :
    CoroutineWorker(appContext, params) {

    private val api: SaversLeagueApi by lazy { Deps.api() }
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }

    override suspend fun doWork(): Result {
        val applicationPackageName = applicationContext.packageName
        val campaignId = sdkConfig.campaignId
        val userId = sdkConfig.installationId

        if (campaignId.isNullOrEmpty() || userId.isEmpty()) {
            SLog.i("$ANALYTICS_TAG Missing campaignID or userId. stopping.")
            return Result.failure()
        }

        val events = Analytics.getTrackedEvents()
        if (events.isEmpty()) {
            return Result.success()
        }

        SLog.i("$ANALYTICS_TAG Sending analytics events [${events.size}]")
        val response = try {
            api.sendAnalyticsEvents(AnalyticsEventsDto(events.map { event ->
                event.toDto(
                    campaignId = campaignId,
                    userId = userId,
                    source = applicationPackageName,
                    appVersion = getAppVersionName(appContext),
                )
            }))
        } catch (e: Exception) {
            SLog.e("$ANALYTICS_TAG Error while sending analytics events. Retrying", e)
            return Result.failure()
        }

        return if (response.status == HttpStatusCode.OK) {
            Analytics.clearTrackedEvents(events)
            Result.success()
        } else {
            SLog.e("$ANALYTICS_TAG Error while sending analytics events: ${response.status}. Retrying")
            Result.failure()
        }
    }

    companion object {
        const val ANALYTICS_TAG = "[analytics]"
        const val NAME = "ANALYTICS_SEND_WORKER"
        private const val RETRY_DELAY = 30L
        private const val INITIAL_DELAY_MINUTES = 1L // so Adjust attribution can be received

        fun createPeriodicWork(): PeriodicWorkRequest {
            val constraints =
                Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

            return PeriodicWorkRequestBuilder<AnalyticsSendWorker>(15, TimeUnit.MINUTES)
                .setInitialDelay(INITIAL_DELAY_MINUTES, TimeUnit.MINUTES)
                .setConstraints(constraints)
                .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, RETRY_DELAY, TimeUnit.SECONDS)
                .build()
        }

        fun createUniqueWorkRequest(): OneTimeWorkRequest {
            val constraints = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build()
            return OneTimeWorkRequestBuilder<AnalyticsSendWorker>()
                .setConstraints(constraints)
                .build()
        }
    }
}

private fun getAppVersionName(context: Context): String {
    return try {
        val packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
        packageInfo.versionName ?: "unknown"
    } catch (e: PackageManager.NameNotFoundException) {
        "unknown"
    }
}