package app.appnomix.sdk.internal.domain

import DomainData
import DomainDataDto
import android.content.Context
import androidx.collection.LruCache
import androidx.room.withTransaction
import app.appnomix.sdk.external.CouponsSdkFacade
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.local.AppnomixDatabase
import app.appnomix.sdk.internal.data.local.DatabaseDestructiveMigrationListener
import app.appnomix.sdk.internal.data.local.DynamicConfig
import app.appnomix.sdk.internal.data.local.dao.ConfigDao
import app.appnomix.sdk.internal.data.local.dao.CouponDao
import app.appnomix.sdk.internal.data.local.dao.DemandDao
import app.appnomix.sdk.internal.data.local.model.CouponWithInteractionEntity
import app.appnomix.sdk.internal.data.local.model.toConfig
import app.appnomix.sdk.internal.data.local.model.toCoupon
import app.appnomix.sdk.internal.data.local.model.toDemand
import app.appnomix.sdk.internal.data.local.model.toEntity
import app.appnomix.sdk.internal.data.network.DataResponse
import app.appnomix.sdk.internal.data.network.SaversLeagueApi
import app.appnomix.sdk.internal.data.network.model.CouponDto
import app.appnomix.sdk.internal.data.network.model.DemandDto
import app.appnomix.sdk.internal.data.network.model.PageInfoDto
import app.appnomix.sdk.internal.data.network.model.toEntity
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.domain.model.Demand
import app.appnomix.sdk.internal.utils.SLog
import extractBaseDomain
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import toDomainData
import java.lang.ref.WeakReference
import java.time.LocalDateTime
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.Executors

class CouponsRepo(
    private val context: Context,
    private val sdkConfig: SdkConfig,
    private val api: SaversLeagueApi
) : DatabaseDestructiveMigrationListener {
    private val singleDispatcher = Executors.newSingleThreadExecutor().asCoroutineDispatcher()
    private val scope = CoroutineScope(singleDispatcher)

    private val db: AppnomixDatabase by lazy { AppnomixDatabase.instance(context, this) }
    private val couponsDao: CouponDao by lazy { db.couponsDao() }
    private val demandsDao: DemandDao by lazy { db.demandsDao() }
    private val configsDao: ConfigDao by lazy { db.configDao() }

    private var couponsCache = CopyOnWriteArrayList<Coupon>()
    private val domainDataLruCache = LruCache<String, DomainDataDto.DomainDataV1Dto>(10)

    private var weakListener: WeakReference<CouponsUpdateListener?>? = null
    private val exceptionHandler = CoroutineExceptionHandler { _, exception ->
        SLog.e("Something went wrong while doing repo ops", exception)
    }

    init {
        populateCache()
        initWithConfig()
    }

    suspend fun hasData(): Boolean {
        return withContext(scope.coroutineContext) {
            configsDao.get() != null && demandsDao.count() > 0
        }
    }

    override fun onDestructiveMigration() {
        SLog.e("Destructive migration has occurred. Triggering data refresh")
        CouponsSdkFacade.refreshData()
    }

    suspend fun fetchHomepageCoupons() {
        if (context.packageName != "com.saversleague.coupons") {
            return
        }

        SLog.i("fetching homepage coupons")
        fetchPagedData(
            itemsTag = "coupons",
            initialFetch = { api.getCoupons(true) },
            nextPageFetch = { url -> api.getNextPageData<CouponDto>(url) },
            processEntities = { data ->
                val filteredEntities = data
                    .filter { LocalDateTime.now() in it.startDate..it.endDate }
                    .map { it.toEntity(true) }

                db.withTransaction {
                    val previousItems = couponsDao.getAll()
                    val updatedNewItems = filteredEntities.map { newItem ->
                        val previousItem = previousItems.find { it.couponEntity.id == newItem.id }
                        val newCouponEntity = newItem.copy(
                            isHomepageCoupon = previousItem?.couponEntity?.isHomepageCoupon == true
                        )
                        val previousSnoozeEntity = previousItem?.interactionEntity?.copy()

                        CouponWithInteractionEntity(
                            couponEntity = newCouponEntity,
                            interactionEntity = previousSnoozeEntity
                        )
                    }
                    couponsDao.replace(updatedNewItems)

                    val coupons = updatedNewItems.map {
                        it.toCoupon(api.getCouponImageUrl(it.couponEntity.brandDomain))
                    }
                    couponsCache.clear()
                    couponsCache.addAll(coupons)
                    weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
                }
            }
        )
    }

    suspend fun fetchDemands() {
        SLog.i("fetching demands")
        fetchPagedData(
            itemsTag = "demands",
            initialFetch = { api.getOnDemandRedirects() },
            nextPageFetch = { url -> api.getNextPageData<DemandDto>(url) },
            processEntities = { data ->
                val entities = data.map { it.toEntity() }
                db.withTransaction {
                    demandsDao.saveDemands(entities)
                }
            }
        )
    }

    private suspend fun <T> fetchPagedData(
        itemsTag: String,
        initialFetch: suspend () -> DataResponse<T, PageInfoDto>,
        nextPageFetch: suspend (String) -> DataResponse<T, PageInfoDto>,
        processEntities: suspend (List<T>) -> Unit
    ) = withContext(singleDispatcher) {
        var totalCount = 0
        try {
            var nextPageUrl: String? = null
            do {
                val dataResponse: DataResponse<T, PageInfoDto> = nextPageUrl?.let {
                    nextPageFetch(it)
                } ?: initialFetch()

                SLog.d("Retrieved items [$itemsTag]: ${dataResponse.data.size}")

                totalCount = dataResponse.data.size
                processEntities(dataResponse.data)

                nextPageUrl = dataResponse.meta?.nextPage
            } while (nextPageUrl?.isNotBlank() == true)

            SLog.i("Total items [$itemsTag] retrieved: $totalCount [countryCode=${sdkConfig.countryCodeOverride}]")
        } catch (t: Throwable) {
            SLog.e("Something went wrong while processing paged data", t)
        }
    }

    private fun populateCache() {
        scope.launch(exceptionHandler) {
            populateCouponsCache()
        }
    }

    private fun initWithConfig() {
        scope.launch(exceptionHandler) {
            configsDao.get()?.let { entity ->
                sdkConfig.updateWithProperties(entity.toConfig())
            }
        }
    }

    private suspend fun populateCouponsCache() {
        val coupons = couponsDao.getAll()
        couponsCache.clear()
        couponsCache.addAll(coupons.map { it.toCoupon(api.getCouponImageUrl(it.couponEntity.brandDomain)) })
        SLog.i("updated coupons cache with: ${coupons.size}")
        weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
    }

    fun storeConfig(dynamicConfig: DynamicConfig) {
        scope.launch(exceptionHandler) {
            configsDao.replace(dynamicConfig.toEntity())
            SLog.i("config stored")
        }
    }

    fun setListener(listener: CouponsUpdateListener) {
        weakListener = WeakReference(listener)
        if (couponsCache.isNotEmpty()) {
            weakListener?.get()?.onCouponsUpdated(couponsCache.toList())
        }
    }

    suspend fun getCheckoutConfig(url: String): DomainData? =
        withContext(singleDispatcher) {
            try {
                val domain = extractBaseDomain(url)
                var cachedData = domainDataLruCache.get(domain)
                if (cachedData == null) {
                    cachedData = api.getBrand(domain) as DomainDataDto.DomainDataV1Dto
                    domainDataLruCache.put(domain, cachedData)
                }

                cachedData.toDomainData(url)
            } catch (e: Exception) {
                SLog.e("something went wrong while fetching brand data", e)
                null
            }
        }

    suspend fun markAsUsed(item: Demand) {
        withContext(singleDispatcher) {
            demandsDao.upsertLastUseTime(domain = item.rootDomain)
        }
    }

    suspend fun getDemands(domain: String): List<Demand> {
        return withContext(singleDispatcher) {
            demandsDao.getActiveForDomainAndCountry(domain = domain)
                .map { it.toDemand(api.getCouponImageUrl(it.demandEntity.rootDomain)) }
        }
    }

    suspend fun hasActiveDemand(domain: String): Boolean {
        return withContext(singleDispatcher) {
            demandsDao.hasActiveDemand(domain = domain)
        }
    }

}

interface CouponsUpdateListener {
    fun onCouponsUpdated(coupons: List<Coupon>)
}
