package app.appnomix.sdk.internal.domain.machine.states

import android.webkit.WebView
import app.appnomix.sdk.internal.ui.PopupDisplay
import app.appnomix.sdk.internal.ui.isLoading
import app.appnomix.sdk.internal.utils.SLog
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import kotlin.coroutines.resume
import kotlin.time.Duration
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds

interface TreeInventory {
    fun contains(type: TreeInventoryType): Boolean
    fun <T> get(type: TreeInventoryType): T?
    fun <T> save(type: TreeInventoryType, data: T)
    fun remove(type: TreeInventoryType)
}

sealed class UiTreeNode : TreeNode()

sealed class TreeNode {
    private val exceptionHandler = CoroutineExceptionHandler { _, exception ->
        SLog.e("[aa] something went wrong while auto-apply graph was executed", exception)
    }
    val mainDispatcher = Dispatchers.Main + SupervisorJob() + exceptionHandler
    private val mainScope = CoroutineScope(mainDispatcher)

    private var transition: TreeTransition = TreeTransition.None
    var inventory: TreeInventory? = null
    var transitionListener: TreeNodeTransitionListener? = null

    fun canTransition(): Boolean {
        return transition != TreeTransition.None
    }

    open fun markReadyForTransition(treeTransition: TreeTransition) {
        transition = treeTransition
        transitionListener?.onReadyForTransition(this, treeTransition)
    }

    fun resume(fromNode: TreeNode? = null) {
        transitionListener?.onResumeRequested(fromNode ?: this)
    }

    open fun pause() {
        performInMainScope {
            inventory?.get<PopupDisplay>(TreeInventoryType.POPUP_DISPLAY)?.apply {
                hide()
            }
        }
    }

    open fun positiveChildNode(): TreeNode = TerminalNode
    open fun negativeChildNode(): TreeNode = TerminalNode

    abstract suspend fun execute()

    suspend inline fun <reified T> evaluateJs(
        webView: WebView,
        jsCode: String,
    ): JsResult<T> =
        withContext(mainDispatcher) {
            SLog.d("evaluating: $jsCode")
            suspendCancellableCoroutine { continuation ->
                webView.evaluateJavascript(jsCode.trim()) { value ->
                    val result = try {
                        when (T::class) {
                            String::class -> JsResult.Success(value as T)
                            Int::class -> JsResult.Success(
                                value.toIntOrNull() as T
                                    ?: throw IllegalArgumentException("Invalid int")
                            )

                            Double::class -> JsResult.Success(
                                value.toDoubleOrNull() as T
                                    ?: throw IllegalArgumentException("Invalid double")
                            )

                            Boolean::class -> JsResult.Success(value.toBoolean() as T)
                            else -> throw IllegalArgumentException("Unsupported type")
                        }
                    } catch (e: Exception) {
                        SLog.e("[aa] eval: $value", e)
                        JsResult.Error(e.message ?: "Error parsing result")
                    }
                    continuation.resume(result)
                }
            }
        }

    suspend fun waitForWebViewFinishedLoading(
        webView: WebView,
        initialDelay: Duration = 500.milliseconds
    ) {
        delay(initialDelay)
        while (webView.isLoading()) {
            delay(1.seconds)
        }
        // some websites are not reporting loading even though elements are still not rendered
        // even though WebView reports it finished loading.
        // so we need to simulate an additional wait
        delay(1.seconds)
    }

    fun performInMainScope(task: suspend (() -> Unit)) {
        mainScope.launch { task() }
    }

    sealed class JsResult<out T> {
        data class Success<out T>(val value: T) : JsResult<T>()
        data class Error(val error: String) : JsResult<Nothing>()
    }
}

enum class TreeInventoryType {
    CONTEXT,
    CURRENT_URL,
    CHECKOUT_CONFIG,
    POPUP_DISPLAY,
    WEB_VIEW,
    CART_INITIAL_TOTAL,
    CART_NEW_TOTAL,
    COUPON_LIST,
    COUPON_INDEX_BEING_VERIFIED,
    BEST_COUPON_INDEX_DETECTED,
    URL_FOLLOWER,
    CURRENCY_SYMBOL,
    EXTERNAL_COUPON_CODE_APPLIER,
    FAB_POSITION_DATA,
}

sealed class TreeTransition {
    data object Positive : TreeTransition()
    data object Negative : TreeTransition()
    data object None : TreeTransition()
}

interface TreeNodeTransitionListener {
    fun onReadyForTransition(node: TreeNode, transition: TreeTransition)
    fun onResumeRequested(node: TreeNode)
}
