package app.appnomix.sdk.internal.domain.machine.states.analytics

import CheckoutPageConfig
import app.appnomix.sdk.internal.analytics.Analytics
import app.appnomix.sdk.internal.analytics.AnalyticsEvent
import app.appnomix.sdk.internal.analytics.AnalyticsName
import app.appnomix.sdk.internal.domain.machine.states.TreeInventoryType
import app.appnomix.sdk.internal.domain.machine.states.TreeNode
import app.appnomix.sdk.internal.domain.model.Coupon

private fun addBrandDomainAnalyticsParam(
    treeNode: TreeNode,
    builder: AnalyticsEvent.Builder
) {
    treeNode.inventory?.get<CheckoutPageConfig>(TreeInventoryType.CHECKOUT_CONFIG)?.brandDomain?.let {
        builder.addParam(AnalyticsName.EventPayloadKey.BRAND_DOMAIN, it)
    }
}

private fun addBrandNameAnalyticsParam(
    treeNode: TreeNode,
    builder: AnalyticsEvent.Builder
) {
    treeNode.inventory?.get<CheckoutPageConfig>(TreeInventoryType.CHECKOUT_CONFIG)?.brandName?.let {
        builder.addParam(AnalyticsName.EventPayloadKey.BRAND_NAME, it)
    }
}

private fun addAvailableCouponsAnalyticsParam(
    treeNode: TreeNode,
    builder: AnalyticsEvent.Builder
) {
    treeNode.inventory?.get<List<Coupon>>(TreeInventoryType.COUPON_LIST)?.let {
        builder.addParam(AnalyticsName.EventPayloadKey.COUPONS_AVAILABLE, it.size.toString())
    }
}

private fun addCouponsTriedAnalyticsParam(
    treeNode: TreeNode,
    builder: AnalyticsEvent.Builder
) {
    treeNode.inventory?.get<Int>(TreeInventoryType.COUPON_INDEX_BEING_VERIFIED)?.let {
        builder.addParam(AnalyticsName.EventPayloadKey.COUPONS_TRIED, (it + 1).toString())
    }
}

private fun addCurrencySymbolAnalytics(
    treeNode: TreeNode,
    builder: AnalyticsEvent.Builder
) {
    treeNode.inventory?.get<String>(TreeInventoryType.CURRENCY_SYMBOL)?.let {
        builder.addParam(AnalyticsName.EventPayloadKey.CURRENCY, it)
    }
}

private fun sendAutoApplyEvent(
    eventType: String,
    builder: (AnalyticsEvent.Builder) -> Unit
) {
    Analytics.trackEvent(
        AnalyticsEvent.Builder(AnalyticsName.EventCategory.AUTO_APPLY).apply {
            this.name = eventType
            builder.invoke(this)
        }.build()
    )
}

internal fun TreeNode.logCheckoutLoadedAnalytics(cartTotal: Double) {
    sendAutoApplyEvent(AnalyticsName.EventType.CHECKOUT_LOADED) {
        addBrandDomainAnalyticsParam(this@logCheckoutLoadedAnalytics, it)
        it.addParam(AnalyticsName.EventPayloadKey.CART_VALUE, cartTotal.toString())
    }
}

internal fun TreeNode.logStartAutoApplyTryLaterAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.TRY_LATER_AUTO_APPLY) {
        addBrandDomainAnalyticsParam(this@logStartAutoApplyTryLaterAnalytics, it)
    }
}

internal fun TreeNode.logStartAutoApplyCloseAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.CLOSE_AUTO_APPLY) {
        addBrandDomainAnalyticsParam(this@logStartAutoApplyCloseAnalytics, it)
    }
}

internal fun TreeNode.logStartAutoApplyAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.START_AUTO_APPLY) { builder ->
        addBrandDomainAnalyticsParam(this@logStartAutoApplyAnalytics, builder)
        addAvailableCouponsAnalyticsParam(this@logStartAutoApplyAnalytics, builder)
    }
}

internal fun TreeNode.logShowAutoApplyAskAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.SHOW_AUTO_APPLY) { builder ->
        addBrandDomainAnalyticsParam(this@logShowAutoApplyAskAnalytics, builder)
        addAvailableCouponsAnalyticsParam(this@logShowAutoApplyAskAnalytics, builder)
    }
}

internal fun TreeNode.logAutoApplyCodeTriedSuccessAnalytics(initialTotal: Double, newTotal: Double) {
    sendAutoApplyEvent(AnalyticsName.EventType.CODE_TRIED_SUCCESS) { builder ->
        addBrandNameAnalyticsParam(this@logAutoApplyCodeTriedSuccessAnalytics, builder)

        inventory?.get<Int>(TreeInventoryType.COUPON_INDEX_BEING_VERIFIED)?.let { couponIndex ->
            inventory?.get<List<Coupon>>(TreeInventoryType.COUPON_LIST)?.get(couponIndex)?.code?.let {
                builder.addParam(AnalyticsName.EventPayloadKey.COUPON_CODE, it)
            }
        }

        builder.addParam(AnalyticsName.EventPayloadKey.SAVINGS, (initialTotal - newTotal).toString())
        builder.addParam(AnalyticsName.EventPayloadKey.CART_VALUE, newTotal.toString())
        addCurrencySymbolAnalytics(this@logAutoApplyCodeTriedSuccessAnalytics, builder)
    }
}

internal fun TreeNode.logAutoApplyCodeTriedFailureAnalytics(initialTotal: Double) {
    sendAutoApplyEvent(AnalyticsName.EventType.CODE_TRIED_FAILURE) { builder ->
        addBrandNameAnalyticsParam(this@logAutoApplyCodeTriedFailureAnalytics, builder)

        inventory?.get<Int>(TreeInventoryType.COUPON_INDEX_BEING_VERIFIED)?.let { couponIndex ->
            inventory?.get<List<Coupon>>(TreeInventoryType.COUPON_LIST)?.get(couponIndex)?.code?.let {
                builder.addParam(AnalyticsName.EventPayloadKey.COUPON_CODE, it)
            }
        }

        builder.addParam(AnalyticsName.EventPayloadKey.SAVINGS, 0.0.toString())
        builder.addParam(AnalyticsName.EventPayloadKey.CART_VALUE, initialTotal.toString())
        addCurrencySymbolAnalytics(this@logAutoApplyCodeTriedFailureAnalytics, builder)
    }
}



internal fun TreeNode.logAutoApplyProgressCloseAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.CANCEL_APPLYING) { builder ->
        addBrandNameAnalyticsParam(this@logAutoApplyProgressCloseAnalytics, builder)
        addAvailableCouponsAnalyticsParam(this@logAutoApplyProgressCloseAnalytics, builder)
        addCouponsTriedAnalyticsParam(this@logAutoApplyProgressCloseAnalytics, builder)
    }
}

internal fun TreeNode.logAutoApplyProgressStopAndApplyAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.STOP_AND_APPLY_SHOW) {
        addBrandNameAnalyticsParam(this@logAutoApplyProgressStopAndApplyAnalytics, it)
        addCouponsTriedAnalyticsParam(this@logAutoApplyProgressStopAndApplyAnalytics, it)
        addAvailableCouponsAnalyticsParam(this@logAutoApplyProgressStopAndApplyAnalytics, it)
    }
}

internal fun TreeNode.logAutoApplyStopAndApplyKeepSearchingAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.STOP_AND_APPLY_KEEP_SEARCHING) {
        addBrandDomainAnalyticsParam(this@logAutoApplyStopAndApplyKeepSearchingAnalytics, it)
    }
}

internal fun TreeNode.logAutoApplyStopAndApplyGoToCartAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.STOP_AND_APPLY_GO_TO_CHECKOUT) {
        addBrandDomainAnalyticsParam(this@logAutoApplyStopAndApplyGoToCartAnalytics, it)
    }
}

internal fun TreeNode.logAutoApplyFinishedWithSavingsAnalytics(initialTotal: Double, newTotal: Double) {
    sendAutoApplyEvent(AnalyticsName.EventType.AUTO_APPLY_FINISHED_WITH_SAVINGS) { builder ->
        addBrandNameAnalyticsParam(this@logAutoApplyFinishedWithSavingsAnalytics, builder)
        builder.addParam(AnalyticsName.EventPayloadKey.SAVINGS, (initialTotal - newTotal).toString())
        builder.addParam(AnalyticsName.EventPayloadKey.CART_TOTAL, newTotal.toString())
        addCurrencySymbolAnalytics(this@logAutoApplyFinishedWithSavingsAnalytics, builder)
    }
}

internal fun TreeNode.logAutoApplyFinishedContinueToCheckoutAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.CONTINUE_TO_CHECKOUT_FINISH) {
        addBrandNameAnalyticsParam(this@logAutoApplyFinishedContinueToCheckoutAnalytics, it)
    }
}

internal fun TreeNode.logAutoApplyNoDiscountShowAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.FINISHED_NO_DISCOUNT) {
        addBrandNameAnalyticsParam(this@logAutoApplyNoDiscountShowAnalytics, it)
    }
}

internal fun TreeNode.logAutoApplyNoDiscountContinueToCheckoutAnalytics() {
    sendAutoApplyEvent(AnalyticsName.EventType.CONTINUE_TO_CHECKOUT_NO_DISCOUNT) {
        addBrandNameAnalyticsParam(this@logAutoApplyNoDiscountContinueToCheckoutAnalytics, it)
    }
}