import android.net.Uri
import app.appnomix.sdk.internal.data.local.model.toCoupon
import app.appnomix.sdk.internal.data.network.model.toEntity
import app.appnomix.sdk.internal.domain.browsers.ensureHttpsPrefix
import app.appnomix.sdk.internal.domain.model.Coupon
import java.net.MalformedURLException
import java.net.URI
import java.net.URL
import kotlin.math.max

data class DomainData(
    val checkoutConfig: CheckoutPageConfig? = null,
    val coupons: List<Coupon> = listOf()
)

data class CheckoutPageConfig(
    val brandDomain: String,
    val brandName: String,
    val imageUrl: String? = null,
    val delayAfterCouponMs: Int? = null,
    val delayBetweenCouponMs: Int? = null,
    val reloadOnApply: Boolean? = null,
    val reloadOnRemove: Boolean? = null,
    val selectorCouponApplyButton: String? = null,
    val selectorCouponError: String? = null,
    val selectorCouponInput: String? = null,
    val selectorCouponRemoveButton: String? = null,
    val selectorClickBeforeStart: String? = null,
    val selectorCouponSuccess: String? = null,
    val selectorOrderTotalElement: String? = null,
)

fun DomainDataDto.DomainDataV1Dto.toDomainData(url: String): DomainData? {
    val domain = extractBaseDomain(url)
    if (rootDomain != domain) return null

    var finalDelayAfterCouponMs: Int = 0
    var finalDelayBetweenCouponMs: Int = 0
    var finalReloadOnApply: Boolean = false
    var finalReloadOnRemove: Boolean = false
    val finalSelectorCouponApplyButton = mutableSetOf<String>()
    val finalSelectorCouponError = mutableSetOf<String>()
    val finalSelectorCouponInput = mutableSetOf<String>()
    val finalSelectorCouponRemoveButton = mutableSetOf<String>()
    val finalSelectorClickBeforeInsert = mutableSetOf<String>()
    val finalSelectorCouponSuccess = mutableSetOf<String>()
    val finalSelectorOrderTotalElement = mutableSetOf<String>()

    val urlPath = getUrlPath(url) ?: ""
    checkoutPages?.filter { it.rootDomain == domain }
        ?.forEach {
            val paths = it.path?.split(",")
            if (paths?.any { urlPath.endsWith(it) } == true) {
                finalDelayAfterCouponMs = max(finalDelayAfterCouponMs, it.delayAfterCouponMs ?: 0)
                finalDelayBetweenCouponMs =
                    max(finalDelayAfterCouponMs, it.delayBetweenCouponMs ?: 0)
                finalReloadOnApply = finalReloadOnApply || (it.reloadOnApply == true)
                finalReloadOnRemove = finalReloadOnRemove || (it.reloadOnRemove == true)
                finalSelectorCouponApplyButton += it.selectorCouponApplyButton ?: ""
                finalSelectorCouponError += it.selectorCouponError ?: ""
                finalSelectorCouponInput += it.selectorCouponInput ?: ""
                finalSelectorCouponRemoveButton += it.selectorCouponRemoveButton ?: ""
                finalSelectorClickBeforeInsert += it.selectorClickBeforeStart ?: ""
                finalSelectorCouponSuccess += it.selectorCouponSuccess ?: ""
                finalSelectorOrderTotalElement += it.selectorOrderTotalElement ?: ""
            }
        }

    val config = CheckoutPageConfig(
        brandDomain = rootDomain,
        brandName = this.brandName ?: "",
        imageUrl = this.image,
        delayAfterCouponMs = finalDelayAfterCouponMs,
        delayBetweenCouponMs = finalDelayBetweenCouponMs,
        reloadOnApply = finalReloadOnApply,
        reloadOnRemove = finalReloadOnRemove,
        selectorCouponApplyButton = finalSelectorCouponApplyButton.joinToString(separator = ","),
        selectorCouponError = finalSelectorCouponError.joinToString(separator = ","),
        selectorCouponInput = finalSelectorCouponInput.joinToString(separator = ","),
        selectorCouponRemoveButton = finalSelectorCouponRemoveButton.joinToString(separator = ","),
        selectorClickBeforeStart = finalSelectorClickBeforeInsert.joinToString(separator = ","),
        selectorCouponSuccess = finalSelectorCouponSuccess.joinToString(separator = ","),
        selectorOrderTotalElement = finalSelectorOrderTotalElement.joinToString(separator = ","),
    )

    val coupons = (this.coupons?.map { it.toEntity(false).toCoupon(this.image ?: "") } ?: listOf())

    return DomainData(
        checkoutConfig = config,
        coupons = coupons
    )
}

private fun getUrlPath(url: String): String? {
    return try {
        URL(url).path
    } catch (e: Exception) {
        if (e is MalformedURLException) {
            return Uri.parse(url).path
        }
        throw e
    }
}

fun extractBaseDomain(url: String): String {
    if (url.isEmpty()) return url
    val cleanedUrl = url.substringBefore("?")
    val uri = URI(cleanedUrl.ensureHttpsPrefix())
    return uri.host?.removePrefix("www.") ?: cleanedUrl
}
