package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.Bitmap
import android.net.http.SslError
import android.util.AttributeSet
import android.view.View
import android.webkit.ConsoleMessage
import android.webkit.GeolocationPermissions
import android.webkit.SslErrorHandler
import android.webkit.WebChromeClient
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebView.setWebContentsDebuggingEnabled
import android.webkit.WebViewClient
import app.appnomix.sdk.BuildConfig
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.domain.machine.AppnomixStateMachine
import app.appnomix.sdk.internal.utils.SLog

class OverlayWebview(context: Context, attrs: AttributeSet?) : WebView(context, attrs) {
    init {
        settings.apply {
            javaScriptEnabled = true
            domStorageEnabled = true
            databaseEnabled = true
            cacheMode = WebSettings.LOAD_DEFAULT
            setSupportZoom(true)
            builtInZoomControls = true
            displayZoomControls = false
            setGeolocationEnabled(true)
            userAgentString = "Mozilla/5.0 (Linux; Android 10; Mobile) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/88.0.4324.93 Mobile Safari/537.36"
            mixedContentMode = WebSettings.MIXED_CONTENT_COMPATIBILITY_MODE
            setWebContentsDebuggingEnabled(BuildConfig.DEBUG)
        }
        setLayerType(View.LAYER_TYPE_HARDWARE, null)
        webViewClient = object : WebViewClient() {
            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                super.onPageStarted(view, url, favicon)
                SLog.d("onPageStarted: $url")
            }

            override fun onPageFinished(view: WebView, url: String?) {
                super.onPageFinished(view, url)
                SLog.d("onPageFinished: $url")
            }
        }

        setWebChromeClient(object : WebChromeClient() {
            override fun onGeolocationPermissionsShowPrompt(
                origin: String?,
                callback: GeolocationPermissions.Callback?
            ) {
                callback?.invoke(origin, true, false)
            }

            override fun onConsoleMessage(consoleMessage: ConsoleMessage?): Boolean {
                // suppress default webview JS > logcat
                return true
            }

            override fun onProgressChanged(view: WebView, newProgress: Int) {
                val isLoading = newProgress < 100
                view.tag = isLoading
            }
        })
    }

    fun click(
        viewIdResourceName: String?,
        contentDescription: CharSequence?,
        text: CharSequence?
    ) {
        val contentDescriptionString =
            contentDescription?.trim()?.toString()?.lowercase() ?: "_NULL_"
        val textString = text?.trim()?.toString()?.lowercase() ?: "_NULL_"
        val idString = viewIdResourceName?.trim()?.lowercase() ?: "_NULL_"

        evaluateJavascript(
            """
            (function clickByMatchers() {
                var elements = document.querySelectorAll(
                    'button, [role="button"], [role="link"], a, input[type="button"], input[type="submit"], input[type="reset"], [tabindex], [onclick], span, div'
                );
                
                var id = `$idString` || "";
                var text1 = `$contentDescriptionString` || "";
                var text2 = `$textString` || "";

                for (var i = 0; i < elements.length; i++) {
                    var element = elements[i];
                    var isClickable = element.hasAttribute('onclick') || 
                      element.hasAttribute('role') ||
                      element.hasAttribute('tabindex') ||
                      element.tagName.toLowerCase() === 'a' ||
                      window.getComputedStyle(element).cursor === 'pointer';
                          
                    if (isClickable && element.id && element.id.trim().toLowerCase() === id && id !== '') {
                        element.click();
                        return element;
                    }
                }

                for (var i = 0; i < elements.length; i++) {
                    var element = elements[i];
                    var isClickable = element.hasAttribute('onclick') || 
                      element.hasAttribute('role') ||
                      element.hasAttribute('tabindex') ||
                      element.tagName.toLowerCase() === 'a' ||
                      window.getComputedStyle(element).cursor === 'pointer';
                    if (!isClickable) continue;
                      
                    var ariaLabel = element.getAttribute("aria-label") || "";
                    var title = element.getAttribute("title") || "";
                    var innerText = element.innerText || "";
                    var outerText = element.outerText || "";
                    
                    var ariaLabelLower = ariaLabel.trim().toLowerCase();
                    var titleLower = title.trim().toLowerCase();
                    var innerTextLower = innerText.trim().toLowerCase();
                    var outerTextLower = outerText.trim().toLowerCase();

                    if (ariaLabelLower === text1 || ariaLabelLower === text2 || 
                        titleLower === text1 || titleLower === text2 || 
                        innerTextLower === text1 || innerTextLower === text2 || 
                        outerTextLower === text1 || outerTextLower === text2) {
                        
                        element.click();
                        return element;
                    }
                }

                return "";
            })();
    """.trimIndent()
        ) {
            val success = it != null && it.isNotEmpty()
            if (success) {
                SLog.d("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) was clicked [$it]")
            } else {
                SLog.d("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) was NOT clicked")
            }
        }
    }
}


fun WebView.isLoading(): Boolean {
    return this.tag as? Boolean == true
}
