package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.drawable.Drawable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.ScrollView
import android.widget.TextView
import androidx.core.view.forEachIndexed
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.domain.model.Coupon
import com.google.android.material.progressindicator.LinearProgressIndicator
import java.time.Instant
import java.time.LocalTime
import java.util.Locale

class TryingCouponsView(
    private val context: Context,
    private val logo: Drawable
) {
    private var overlay: View? = null
    private var logoImageView: ImageView? = null
    private var titleTextView: TextView? = null
    private var descriptionTextView: TextView? = null
    private var laterButton: Button? = null
    private var couponsScrollView: ScrollView? = null
    private var couponsLayout: ViewGroup? = null
    private var progressBar: LinearProgressIndicator? = null

    fun create(
        data: PopupViewData,
        dismissAction: () -> Unit,
    ): View {

        overlay = LayoutInflater.from(context).inflate(R.layout.trying_coupons, null)

        logoImageView = overlay?.findViewById(R.id.logo_imageView)
        titleTextView = overlay?.findViewById(R.id.title_textView)
        descriptionTextView = overlay?.findViewById(R.id.description_textView)
        laterButton = overlay?.findViewById(R.id.later_button)
        couponsScrollView = overlay?.findViewById(R.id.coupons_scrollView)
        couponsLayout = overlay?.findViewById(R.id.coupons_layout)
        progressBar = overlay?.findViewById(R.id.progress_bar)

        logoImageView?.setImageDrawable(logo)

        laterButton?.setOnClickListener { dismissAction() }

        configure(data = data)

        return overlay as View
    }

    fun configure(data: PopupViewData) {
        val couponList =
            data.get<List<Coupon>>(PopupViewData.PopupViewDataType.COUPON_LIST)
                ?: listOf()
        val currentCouponIndex =
            data.get(PopupViewData.PopupViewDataType.CURRENT_COUPON_INDEX_BEING_VALIDATED)
                ?: -1
        val currencySymbol =
            data.get<String>(PopupViewData.PopupViewDataType.CURRENCY_SYMBOL) ?: ""

        if (couponsLayout?.childCount != couponList.size) {
            couponsLayout?.removeAllViews()
            couponList.forEach { item ->
                val itemLayout =
                    LayoutInflater.from(context).inflate(R.layout.coupon_item, null)
                itemLayout.apply {
                    val couponDescription =
                        findViewById<TextView>(R.id.couponDescription_textView)
                    val couponCode = findViewById<TextView>(R.id.couponCode_textView)

                    couponDescription.text = item.description
                    couponCode.text = item.code
                    itemLayout.setOnClickListener {
                        item.code.copyToClipboard(context)
                    }
                }
                couponsLayout?.addView(itemLayout)
            }
        }
        val currentProgress =
            (currentCouponIndex + 1).times(100).div(couponList.size).coerceIn(0..100)

        data.get<Double>(PopupViewData.PopupViewDataType.TOTAL_SAVINGS)?.let {
            if (it > 0) {
                laterButton?.text = context.resources.getString(R.string.stop_and_apply)
            }

            val savingsValue = String.format(Locale.getDefault(), "%.2f", it)
            titleTextView?.setText(
                context.resources.getString(
                    when {
                        LocalTime.now().minute % 7 == 0 -> R.string.trying_you_have_saved_1
                        else -> R.string.trying_you_have_saved_2
                    },
                    "$currencySymbol$savingsValue"
                )
            )
        } ?: run {
            titleTextView?.setText(
                when {
                    currentProgress < 20 -> R.string.trying_1
                    currentProgress < 40 -> R.string.trying_2
                    currentProgress < 60 -> R.string.trying_3
                    currentProgress < 80 -> R.string.trying_4
                    else -> R.string.trying_5
                }
            )
        }

        descriptionTextView?.text = context.getString(
            R.string.testing_out_of_coupons_codes,
            currentCouponIndex + 1,
            couponList.size
        ).boldNumbers()

        couponsLayout?.forEachIndexed { index, view ->
            if (currentCouponIndex == index) {
                couponsScrollView?.smoothScrollTo(view.x.toInt(), view.y.toInt())
                view.alpha = 1f
            } else {
                view.alpha = 0.5f
            }
        }

        progressBar?.progress = currentProgress
    }
}
