package app.appnomix.sdk.internal.ui.onboarding

import android.content.Context
import androidx.annotation.ColorRes
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import app.appnomix.sdk.R

data class OnboardingScreenData(
    val content: OnboardingContent,
    @StringRes val titleRes: Int?,
    @StringRes val mainButtonTitleRes: Int,
    @StringRes val secondaryButtonTitleRes: Int?,
    @ColorRes val secondaryButtonTextColorRes: Int? = null,
    @DrawableRes val secondaryButtonDrawableRes: Int = R.drawable.transparent_bg,
) {
    fun contentWithAppName(
        @StringRes stringRes: Int,
        context: Context
    ): String {
        return context.getString(stringRes, context.getString(context.applicationInfo.labelRes))
    }
}

sealed class OnboardingContent {
    data class ImageText(
        @DrawableRes val imageDrawableRes: Int,
        @StringRes val textRes: Int,
    ) : OnboardingContent()

    data class FirstAccessibility(
        @StringRes val textRes: Int,
    ) : OnboardingContent()

    data class SecondAccessibility(
        @StringRes val titleRes: Int,
        @StringRes val useAppRes: Int,
        @StringRes val appShortcutRes: Int,
    ) : OnboardingContent()

    data class TextWithMore(
        @StringRes val textRes: Int,
    ) : OnboardingContent()

    data class Image(
        @DrawableRes val imageDrawableRes: Int,
    ) : OnboardingContent()
}

enum class OnboardingScreenType(val order: Int) {
    GET_STARTED_1(0),
    ACCESSIBILITY_PERMISSION_EXPLAINER(1),
    ACCESSIBILITY_PERMISSION_STEP_1(2),
    ACCESSIBILITY_PERMISSION_STEP_2(3),
    SETUP_DONE(4),
}

val ONBOARDING_SCREENS = mapOf(
    OnboardingScreenType.GET_STARTED_1 to OnboardingScreenData(
        titleRes = R.string.onboarding_title_1,
        content = OnboardingContent.ImageText(
            imageDrawableRes = R.drawable.onboarding_first,
            textRes = R.string.onboarding_description_1
        ),
        mainButtonTitleRes = R.string.get_started,
        secondaryButtonTitleRes = null
    ),

    OnboardingScreenType.ACCESSIBILITY_PERMISSION_EXPLAINER to OnboardingScreenData(
        titleRes = R.string.accessibility_permission,
        content = OnboardingContent.TextWithMore(
            textRes = R.string.accessibility_permission_explainer
        ),
        mainButtonTitleRes = R.string.agree,
        secondaryButtonTitleRes = R.string.disagree,
        secondaryButtonTextColorRes = R.color.sdk_primary,
        secondaryButtonDrawableRes = R.drawable.transparent_bg_bordered_button
    ),

    OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1 to OnboardingScreenData(
        titleRes = R.string.activate_app,
        content = OnboardingContent.FirstAccessibility(
            textRes = R.string.accessibility_walk_1
        ),
        mainButtonTitleRes = R.string.next,
        secondaryButtonTitleRes = R.string.skip_for_now,
        secondaryButtonTextColorRes = R.color.sdk_primary,
    ),

    OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2 to OnboardingScreenData(
        titleRes = R.string.activate_app,
        content = OnboardingContent.SecondAccessibility(
            titleRes = R.string.accessibility_walk_2,
            useAppRes = R.string.use_app,
            appShortcutRes = R.string.app_shortcut
        ),
        mainButtonTitleRes = R.string.enable,
        secondaryButtonTitleRes = R.string.skip_for_now,
        secondaryButtonTextColorRes = R.color.sdk_primary,
    ),

    OnboardingScreenType.SETUP_DONE to OnboardingScreenData(
        titleRes = R.string.setup_complete,
        content = OnboardingContent.Image(
            imageDrawableRes = R.drawable.onboarding_first,
        ),
        mainButtonTitleRes = R.string.start_saving,
        secondaryButtonTitleRes = null
    ),
)