package app.appnomix.sdk.external

import android.accessibilityservice.AccessibilityServiceInfo
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.provider.Settings
import android.view.accessibility.AccessibilityManager
import androidx.work.ExistingPeriodicWorkPolicy
import app.appnomix.sdk.internal.CouponsAccessibilityService
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.network.AppnomixConfigSyncWorker
import app.appnomix.sdk.internal.domain.CouponsUpdateListener
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.domain.model.toExternalCoupon
import app.appnomix.sdk.internal.ui.onboarding.OnboardingActivity
import app.appnomix.sdk.internal.utils.SLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference
import java.util.concurrent.TimeUnit

object CouponsSdkFacade : CouponsUpdateListener, AppnomixEventListener {
    private val couponsRepo by lazy { Deps.repo() }
    private val sdkConfig by lazy { Deps.sdkConfig() }
    private val workManager by lazy { Deps.workManager() }
    private val scope = CoroutineScope(Dispatchers.Main)
    private var weakCouponsReadyListener: WeakReference<ExternalCouponsReadyListener?>? = null
    private var weakEventListener: WeakReference<AppnomixEventListener?>? = null
    private lateinit var appContext: Context

    var countryCodeOverride: String?
        get() = sdkConfig.countryCodeOverride
        set(value) {
            sdkConfig.countryCodeOverride = value
        }

    fun initialize(context: Context) {
        appContext = context
    }

    fun setup(config: Config) {
        sdkConfig.authToken = config.authToken
        sdkConfig.clientId = config.clientId
        scope.launch {
            scheduleCouponsSync(keepExisting = couponsRepo.hasData())
        }
    }

    fun refreshData() {
        scheduleCouponsSync(keepExisting = false)
    }

    private fun scheduleCouponsSync(keepExisting: Boolean) {
        if (workManager == null) {
            SLog.e("Cannot schedule as workManager is null")
            return
        }

        try {
            if (sdkConfig.authToken.isNotBlank() && sdkConfig.clientId.isNotBlank()) {
                val configWorker =
                    AppnomixConfigSyncWorker.create(TimeUnit.MILLISECONDS.toHours(sdkConfig.dataRefreshPeriodMillis))

                workManager?.enqueueUniquePeriodicWork(
                    AppnomixConfigSyncWorker.NAME,
                    if (keepExisting) ExistingPeriodicWorkPolicy.KEEP else ExistingPeriodicWorkPolicy.REPLACE,
                    configWorker
                )
            } else {
                SLog.e("Cannot schedule data sync, as sdk isn't fully initialized (missing config auth/clientId)")
            }
        } catch (t: Throwable) {
            SLog.e("Something went wrong during init", t)
        }
    }

    fun onExternalMarketingCampaignKeyReceived(key: String) {
        sdkConfig.saveExternalMarketingCampaignKey(key)
    }

    fun unmetRequirements(): List<SdkRequirement> {
        val isAccessibilityServiceEnabled = isAccessibilityServiceEnabled()

        return mutableListOf<SdkRequirement>().apply {
            if (!isAccessibilityServiceEnabled)
                add(SdkRequirement.ACCESSIBILITY)
        }
    }

    fun isAccessibilityServiceEnabled(): Boolean {
        val accessibilityManager =
            appContext.getSystemService(Context.ACCESSIBILITY_SERVICE) as AccessibilityManager

        val enabledServices = accessibilityManager.getEnabledAccessibilityServiceList(
            AccessibilityServiceInfo.FEEDBACK_ALL_MASK
        )

        for (enabledService in enabledServices) {
            val serviceInfo = enabledService.resolveInfo.serviceInfo
            if (serviceInfo.packageName == appContext.packageName && serviceInfo.name == CouponsAccessibilityService::class.java.name) {
                return true
            }
        }

        return false
    }

    fun registerCouponsReadyListener(listener: ExternalCouponsReadyListener) {
        weakCouponsReadyListener = WeakReference(listener)
        couponsRepo.setListener(this)
    }

    override fun onCouponsUpdated(coupons: List<Coupon>) {
        weakCouponsReadyListener?.get()?.onCouponsAvailable(coupons
            .filter { !it.isForHomepage }
            .map { it.toExternalCoupon() })
    }

    fun launchSdkOnboardingActivity(activity: Activity) {
        if (!sdkConfig.onboardingFinished) {
            val intent = Intent(activity, OnboardingActivity::class.java)
            activity.startActivity(intent)
        } else if (!isAccessibilityServiceEnabled()) {
            goToToAccessibilityOnboarding(activity)
        } else {
            SLog.i("onboarding already finished. no need to relaunch.")
        }
    }

    fun goToToAccessibilityOnboarding(activity: Activity) {
        val intent = Intent(activity, OnboardingActivity::class.java)
        intent.action = OnboardingActivity.TO_ACCESSIBILITY_STEP_ACTION
        activity.startActivity(intent)
    }

    fun goToToAccessibilitySettings(activity: Activity) {
        val intent = Intent(Settings.ACTION_ACCESSIBILITY_SETTINGS)
        activity.startActivity(intent)
    }

    fun registerEventListener(listener: AppnomixEventListener) {
        weakEventListener = WeakReference(listener)
    }

    override fun onAppnomixEvent(event: AppnomixEvent) {
        SLog.i("Sending Appnomix event: $event")
        weakEventListener?.get()?.onAppnomixEvent(event)
    }

    data class Config(
        val authToken: String,
        val clientId: String
    )
}

interface ExternalCouponsReadyListener {
    fun onCouponsAvailable(coupons: List<ExternalCoupon>)
}
