package app.appnomix.sdk.internal.data.network

import android.content.Context
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.NetworkType
import androidx.work.PeriodicWorkRequest
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkerParameters
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.local.toDynamicConfig
import app.appnomix.sdk.internal.domain.CouponsRepo
import app.appnomix.sdk.internal.utils.SLog
import java.util.concurrent.TimeUnit

class AppnomixConfigSyncWorker(
    appContext: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(appContext, workerParams) {
    private val repo: CouponsRepo by lazy { Deps.repo() }
    private val api: SaversLeagueApi by lazy { Deps.api() }
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }

    override suspend fun doWork(): Result {
        if (runAttemptCount >= MAX_RETRY_COUNT) {
            return Result.failure()
        }
        try {
            fetchConfig()
        } catch (e: Throwable) {
            SLog.e("Something went wrong with $NAME", e)
            return Result.retry()
        }
        scheduleDataSync()
        return Result.success()
    }

    private suspend fun fetchConfig() {
        SLog.i("fetching config")
        val configContent = api.getConfig()
        SLog.i("config fetched")
        SLog.i("saving config")
        val dynamicConfig = configContent.toDynamicConfig()
        sdkConfig.updateWithProperties(dynamicConfig)
        repo.storeConfig(dynamicConfig)
    }

    private fun scheduleDataSync() {
        val couponsWorker = AppnomixCouponsSyncWorker.create()
        val demandsWorker = AppnomixDemandsSyncWorker.create()

        Deps.workManager()?.let { workManager ->
            workManager.enqueue(couponsWorker)
            workManager.enqueue(demandsWorker)
        }
    }

    companion object {
        const val MAX_RETRY_COUNT = 5
        const val NAME = "config-sync-worker"
        fun create(repeatIntervalHours: Long): PeriodicWorkRequest {
            val constraints =
                Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
            return PeriodicWorkRequestBuilder<AppnomixConfigSyncWorker>(
                repeatIntervalHours,
                TimeUnit.HOURS
            )
                .setConstraints(constraints)
                .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 30, TimeUnit.SECONDS)
                .build()
        }
    }
}
