package app.appnomix.sdk.internal.domain.machine.states

import android.content.Context
import app.appnomix.sdk.internal.analytics.Analytics
import app.appnomix.sdk.internal.analytics.AnalyticsEvent
import app.appnomix.sdk.internal.analytics.AnalyticsName
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.ui.PopupDisplay
import app.appnomix.sdk.internal.ui.PopupViewData
import app.appnomix.sdk.internal.ui.PopupViewType
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.ui.FabPositionData

data object ShowCopyCodeNode : UiTreeNode() {

    override suspend fun execute() {
        performInMainScope {
            val popupDisplay = inventory?.get<PopupDisplay>(TreeInventoryType.POPUP_DISPLAY)
            val couponList = inventory?.get<List<Coupon>>(TreeInventoryType.COUPON_LIST) ?: listOf()

            Analytics.trackEvent(
                AnalyticsEvent.Builder(
                    eventCategory = AnalyticsName.EventCategory.MONETIZATION
                ).apply { name = AnalyticsName.EventType.COPY_CODE_SHOW }.build()
            )
            showCouponFoundView(popupDisplay = popupDisplay, couponList = couponList)
        }
    }

    private fun showFabView(
        popupDisplay: PopupDisplay?,
        couponList: List<Coupon>,
        code: String = ""
    ) {
        val context = inventory?.get<Context>(TreeInventoryType.CONTEXT)
        val description =
            if (code.isEmpty())
                context?.getString(R.string.tap_to_save)
            else
                context?.getString(R.string.code_copied_to_clipboard)

        val fabPositionData = inventory?.get<FabPositionData>(TreeInventoryType.FAB_POSITION_DATA) ?: FabPositionData()

        popupDisplay?.show(
            viewType = PopupViewType.FAB_MESSAGE,
            onConfirmAction = {
                showCouponFoundView(popupDisplay = popupDisplay, couponList = couponList)
            },
            onDismissAction = {
                inventory?.save(TreeInventoryType.FAB_POSITION_DATA, fabPositionData)
            },
            data = PopupViewData().apply {
                set(PopupViewData.PopupViewDataType.PRIMARY_MESSAGE, code)
                set(
                    PopupViewData.PopupViewDataType.SECONDARY_MESSAGE,
                    description ?: ""
                )
                set(PopupViewData.PopupViewDataType.FAB_POSITION_DATA, fabPositionData)
            }
        )
    }

    private fun showCouponFoundView(
        popupDisplay: PopupDisplay?,
        couponList: List<Coupon>
    ) {
        popupDisplay?.show(
            viewType = PopupViewType.COUPONS_FOUND,
            onConfirmAction = {
                val code = (it as? String) ?: ""
                showFabView(
                    code = code,
                    popupDisplay = popupDisplay,
                    couponList = couponList
                )
                markReadyForTransition(TreeTransition.Positive)
            },
            onDismissAction = {
                showFabView(
                    popupDisplay = popupDisplay,
                    couponList = couponList
                )
            },
            data = PopupViewData().apply {
                set(PopupViewData.PopupViewDataType.COUPON_LIST, couponList)
            }
        )
    }

    override fun pause() {
        performInMainScope {
            inventory?.get<PopupDisplay>(TreeInventoryType.POPUP_DISPLAY)?.apply {
                when (activePopupType()) {
                    PopupViewType.COUPONS_FOUND, PopupViewType.FAB_MESSAGE -> {
                        hide()
                        markReadyForTransition(TreeTransition.Negative)
                    }

                    else -> {}
                }
            }
        }
    }
}
