package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.PixelFormat
import android.graphics.Rect
import android.os.Build
import android.view.LayoutInflater
import android.view.View
import android.view.WindowInsets
import android.view.WindowManager
import android.webkit.WebView
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.domain.browsers.ensureHttpsPrefix
import app.appnomix.sdk.internal.utils.SLog

const val BACKGROUND_WEBVIEW_ENABLED = true

internal class WebviewWindow(private val context: Context) {
    private val windowManager: WindowManager by lazy {
        context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    }
    private var webView: OverlayWebview? = null
    private var lastBounds: Rect? = null
    private var lastUrl: String? = null
    private var windowInsets: WindowInsets? = null

    fun getWebView(): WebView? = webView

    fun update(
        url: String,
        bounds: Rect?
    ) {
        if (!BACKGROUND_WEBVIEW_ENABLED) return
        if (url.isEmpty()) return
        if (bounds == lastBounds && url == lastUrl) return
        if (bounds == null) {
            if (webView?.parent != null)
                windowManager.removeView(webView)
            return
        }
        if (webView == null) {
            webView =
                LayoutInflater.from(context).inflate(R.layout.webview, null) as? OverlayWebview
            webView?.alpha = 0f
        }

        val topOffset = windowInsets?.topInset ?: 0
        val bottomOffset = windowInsets?.bottomInset ?: 0

        val params = WindowManager.LayoutParams(
            bounds.width(),
            bounds.height(),
            bounds.left,
            bounds.top - topOffset - bottomOffset,
            WindowManager.LayoutParams.TYPE_ACCESSIBILITY_OVERLAY,
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
                    or WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE,
            PixelFormat.TRANSLUCENT
        )

        if (webView?.parent != null) {
            windowManager.removeView(webView)
        }

        webView?.setOnApplyWindowInsetsListener { view, insets ->
            if (windowInsets == null || (insets.topInset > 0 || insets.bottomInset > 0))
                windowInsets = insets
            insets
        }
        windowManager.addView(webView, params)

        webView?.loadUrl(url.ensureHttpsPrefix())
        lastUrl = url
        lastBounds = bounds
    }

    fun click(
        viewIdResourceName: String?,
        contentDescription: CharSequence?,
        text: CharSequence?,
        className: CharSequence?,
    ) {
        if (!BACKGROUND_WEBVIEW_ENABLED) return
        webView?.click(
            viewIdResourceName = viewIdResourceName,
            contentDescription = contentDescription,
            text = text
        )
    }

    fun updateScroll(x: Int, y: Int) {
        if (!BACKGROUND_WEBVIEW_ENABLED) return
        SLog.d("scroll to: x=$x y=$y")
        webView?.scrollTo(x, y)
    }

    fun hide() {
        webView?.visibility = View.GONE
    }
}

@Suppress("DEPRECATION")
val WindowInsets.topInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).top
    } else {
        this.systemWindowInsetTop
    }

@Suppress("DEPRECATION")
val WindowInsets.bottomInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).bottom
    } else {
        this.systemWindowInsetBottom
    }
