package app.peac.core.service.rpa;

import app.peac.core.model.rpa.RpaDriverConfig;
import app.peac.core.utils.rpa.driver.ChromeDriverUtils;
import app.peac.core.utils.rpa.driver.EdgeDriverUtils;
import app.peac.core.utils.rpa.driver.FirefoxDriverUtils;
import app.peac.core.utils.rpa.driver.SafariDriverUtils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.openqa.selenium.*;
import org.openqa.selenium.interactions.Actions;
import org.openqa.selenium.remote.UnreachableBrowserException;
import org.openqa.selenium.support.ui.ExpectedCondition;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.TimeUnit;

@Getter
@Slf4j
public class RpaDriver implements AutoCloseable {

    private static final String JS_SCRIPT_TO_GET_SCROLL_HEIGHT = "return document.body.scrollHeight;";

    private static final Random random = new Random();

    private final RpaDriverConfig config;

    @Getter
    private final WebDriver driver;

    @Getter
    private final Actions actions;

    public RpaDriver(RpaDriverConfig config) {
        if (Objects.isNull(config)) {
            throw new IllegalArgumentException("Configuration of RpaDriver shouldn't be null");
        }

        log.info("Creating new RpaDriver for {}", config.getDriverType().name());

        this.config = config;

        switch (config.getDriverType()) {
            case EDGE -> driver = EdgeDriverUtils.createEdgeDriver(config);
            case FIREFOX -> driver = FirefoxDriverUtils.createFirefoxDriver(config);
            case SAFARI -> driver = SafariDriverUtils.createSafariDriver();
            default -> driver = ChromeDriverUtils.createChromeDriver(config);
        }

        log.info("{} RpaDriver created, adding additional settings", config.getDriverType().name());

        driver.manage().timeouts()
                .implicitlyWait(Duration.ofSeconds(10))
                .pageLoadTimeout(Duration.ofMinutes(1))
                .scriptTimeout(Duration.ofMinutes(1));
        driver.manage().window().maximize();

        actions = new Actions(driver);

        log.info("{} RpaDriver is ready", config.getDriverType().name());
    }

    public void openBrowserOn(String url) {
        var counter = 0;
        var maxWaitInSec = driver.manage().timeouts().getPageLoadTimeout().getSeconds();

        while (counter < maxWaitInSec) {
            try {
                driver.get(url);
                break;
            } catch (UnreachableBrowserException e) {
                log.warn("Restarting browser due to exception: {}", e.getMessage());
                waitForSec(1L);
                counter++;
            }
        }
    }

    public void waitForMilliSec(long ms) {
        waitFor(TimeUnit.MILLISECONDS, ms);
    }

    public void waitForSec(long seconds) {
        waitFor(TimeUnit.SECONDS, seconds);
    }

    public void waitFor(TimeUnit unit, long amount) {
        try {
            unit.sleep(amount);
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
        }
    }

    public void imitateUserWait() {
        waitForMilliSec(random.nextLong(1001L, 1999L));
    }

    /**
     * Finds the first element matching given descriptor
     *
     * @param selector any selector like: By.id(), By.name(), By.cssSelector("[CLASS:Edit; INSTANCE:1]")
     * @return {@link WebElement}
     */
    public WebElement findBy(By selector) {
        return driver.findElement(selector);
    }

    public WebElement findBy(ExpectedCondition<WebElement> condition) {
        return findBy(Duration.ofMinutes(1L), condition);
    }

    public WebElement findBy(Duration duration, ExpectedCondition<WebElement> condition) {
        return new WebDriverWait(driver, duration).until(condition);
    }

    public Boolean check(ExpectedCondition<Boolean> condition) {
        return new WebDriverWait(driver, Duration.ofMinutes(1)).until(condition);
    }

    /**
     * Find all elements matching given descriptor. Method returns a UiElementCollection which is a list of SelenideElement
     * objects that can be iterated, and at the same time is an implementation of SelenideElement interface, meaning that you
     * can call methods .sendKeys(), click() etc. on it.
     *
     * @param selector any selector like: By.id(), By.name(), By.cssSelector("[CLASS:Edit; INSTANCE:1]")
     * @return list with UiElement objects or empty list if element was no found
     */
    public List<WebElement> findAllBy(By selector) {
        return driver.findElements(selector);
    }

    public List<WebElement> findAllBy(ExpectedCondition<List<WebElement>> condition) {
        return findAllBy(Duration.ofMinutes(1L), condition);
    }

    public List<WebElement> findAllBy(Duration duration, ExpectedCondition<List<WebElement>> condition) {
        return new WebDriverWait(driver, duration).until(condition);
    }

    public void clickOnElemBy(By locator) {
        findBy(ExpectedConditions.elementToBeClickable(locator)).click();
    }

    public void sendKeysBy(By locator, String value) {
        findBy(ExpectedConditions.elementToBeClickable(locator)).sendKeys(value);
    }

    private void sendKeys(Object text) {
        actions.sendKeys((CharSequence) text);
    }

    public void pressKey(Keys key) {
        sendKeys(key);
    }

    public void pressKeys(Keys... keys) {
        sendKeys(Keys.chord(keys));
    }

    public void scrollToPageBottom() {
        executeJS("window.scrollTo(0,document.body.scrollHeight);");
    }

    public void scrollToPageBottomWithRefreshCheck() {
        long currentHeight;

        do {
            currentHeight = executeJS(JS_SCRIPT_TO_GET_SCROLL_HEIGHT);
            scrollToPageBottom();
            imitateUserWait();
        } while (currentHeight != (Long) executeJS(JS_SCRIPT_TO_GET_SCROLL_HEIGHT));

        //Additional check
        scrollToPageBottom();
        imitateUserWait();

        if (currentHeight != (Long) executeJS(JS_SCRIPT_TO_GET_SCROLL_HEIGHT)) {
            scrollToPageBottomWithRefreshCheck();
        }
    }

    public WebElement moveToElement(By locator) {
        WebElement elem = findBy(locator);
        actions.moveToElement(elem).perform();
        return elem;
    }

    public void refreshPage() {
        driver.navigate().refresh();
    }

    /**
     * Executes JavaScript code on RPA node.
     *
     * @param script script code as a String
     * @return One of Boolean, Long, String, List or SelenideElement. Or null.
     */
    public <T> T executeJS(String script) {
        return executeJS(script, false);
    }

    /**
     * Executes JavaScript code on RPA node with ability to turn off logging of expected errors
     *
     * @param script script code as a String
     * @return One of Boolean, Long, String, List or SelenideElement. Or null.
     */
    @SuppressWarnings("unchecked")
    public <T> T executeJS(String script, boolean enableLog) {
        String logMsg = "Execute Java Script:\r\n" + script.substring(0, Math.min(1000, script.length()));
        try {
            if (enableLog && log.isDebugEnabled()) {
                log.debug(logMsg);
            }
            return (T) ((JavascriptExecutor) driver).executeScript(script);
        } catch (Exception e) {
            if (enableLog) {
                log.error(logMsg, e);
            }
            throw e;
        }
    }

    @Override
    public void close() {
        try {
            driver.quit();
        } catch (IllegalStateException e) {
            log.error("Driver cannot be closed: {}", e.getMessage());
        }
    }
}
