package app.pivo.android.lightsdk.command.impl

import app.pivo.android.lightsdk.command.CommandGenerator
import app.pivo.android.lightsdk.model.Brightness
import app.pivo.android.lightsdk.model.LightPattern

/**
 * Created by murodjon on 2021/08/27
 */
internal class LightCmdGeneratorImpl : CommandGenerator {

    override fun getDeviceInfoBytes() =
        byteArrayOf(0x05, 0xCF.toByte(), 0x03.toByte(), 0x00, 0x00, 0x00)

    override fun getBatteryLevelBytes() =
        byteArrayOf(0x05, 0xCF.toByte(), 0x04, 0x00, 0x00, 0x00)

    override fun getChangeNameBytes(name: String): ByteArray? {
        // 5 elements are needed for changing name
        val length = name.length
        // final byte array size for changing name is 5+(length of name)
        val data = ByteArray(length + 5)
        data[0] = 0x05.toByte()
        data[1] = 0xCF.toByte()
        data[2] = 0x7F.toByte()
        data[3] = length.toByte()
        data[4] = 0x00.toByte()

        for (i in name.indices) {
            data[5 + i] = name[i].code.toByte()
        }
        return data
    }

    override fun getMacAddressBytes() = byteArrayOf(0x05, 0xCF.toByte(), 0x1B, 0x00, 0x00, 0x00)

    override fun getSerialNumberBytes(): ByteArray = byteArrayOf(
        0x05, 0xCF.toByte(), 0x1C, 0x00, 0x00, 0x00
    )

    override fun getStopLightPresetBytes() = byteArrayOf(
        0x05.toByte(),
        0xCF.toByte(),
        0x30.toByte(),
        0x00.toByte(),
        0x00.toByte(),
        0x00.toByte(),
        0x00.toByte()
    )

    override fun getRunLightPresetBytes(
        brightness: Brightness,
        interval: Double,
        patternType: LightPattern
    ): ByteArray? {
        return if (!brightness.isValidBrightnessLevel()) {
            null
        } else if (!intervalMap.containsKey(interval)) {
            null
        } else {
            intervalMap[interval]?.let {
                byteArrayOf(
                    0x05.toByte(), 0xCF.toByte(), 0x31.toByte(), 0x00.toByte(), when (patternType) {
                        LightPattern.Type_1 -> 0x01
                        LightPattern.Type_2 -> 0x02
                        LightPattern.Type_3 -> 0x03
                    },
                    it, brightness.warm.toByte(), brightness.cool.toByte()
                )
            }
        }
    }

    private val intervalMap = mapOf(
        0.1 to 0x01,
        0.2 to 0x02,
        0.3 to 0x03,
        0.4 to 0x04,
        0.5 to 0x05,
        0.6 to 0x06,
        0.7 to 0x07,
        0.8 to 0x08,
        0.9 to 0x09,
        1.0 to 0x0A.toByte(),
        2.0 to 0x0B.toByte(),
        3.0 to 0x0C.toByte(),
        4.0 to 0x0D.toByte(),
        5.0 to 0x0E.toByte(),
        6.0 to 0x0F.toByte(),
        7.0 to 0x10.toByte(),
        8.0 to 0x11.toByte(),
        9.0 to 0x12.toByte(),
        10.0 to 0x13.toByte(),
    )

    override fun getBrightnessLevelBytes() =
        byteArrayOf(0x05.toByte(), 0xCF.toByte(), 0x24.toByte(), 0x00.toByte(), 0x00.toByte(), 0x00.toByte())

    override fun getTurnOffLightBytes() =
        byteArrayOf(0x05.toByte(), 0xCF.toByte(), 0x20.toByte(), 0x00.toByte(), 0x00.toByte(), 0x00.toByte())

    override fun getTurnOnLightBytes(brightness: Brightness): ByteArray? {
        return if (brightness.isValidBrightnessLevel()) {
            byteArrayOf(
                0x05.toByte(),
                0xCF.toByte(),
                0x21.toByte(),
                0x00.toByte(),
                brightness.warm.toByte(),
                brightness.cool.toByte()
            )
        } else null
    }

    override fun getDimSmoothlyBytes(brightness: Brightness): ByteArray? {
        return if (brightness.isValidBrightnessLevel()) {
            byteArrayOf(
                0x05.toByte(),
                0xCF.toByte(),
                0x23.toByte(),
                0x00.toByte(),
                brightness.warm.toByte(),
                brightness.cool.toByte()
            )
        } else null
    }

    override fun getFlash1XBytes(brightness: Brightness): ByteArray? {
        return if (brightness.isValidBrightnessLevel()) {
            byteArrayOf(
                0x05.toByte(),
                0xCF.toByte(),
                0x22.toByte(),
                0x00.toByte(),
                brightness.warm.toByte(),
                brightness.cool.toByte()
            )
        } else null
    }
}