package app.pivo.android.lightsdk.controller.light

import android.content.Context
import app.pivo.android.lightsdk.command.CommandGenerator
import app.pivo.android.lightsdk.command.impl.LightCmdGeneratorImpl
import app.pivo.android.lightsdk.controller.ble.BluetoothController
import app.pivo.android.lightsdk.controller.ble.BluetoothControllerCallback
import app.pivo.android.lightsdk.controller.ble.BluetoothControllerImpl
import app.pivo.android.lightsdk.controller.ble.LightDevice
import app.pivo.android.lightsdk.model.Brightness
import app.pivo.android.lightsdk.model.DeviceInfo
import app.pivo.android.lightsdk.model.LightPattern
import app.pivo.android.lightsdk.reply.ReplyHandler
import app.pivo.android.lightsdk.uuid.PivoDeviceUUIDs

/**
 * Created by murodjon on 2021/08/27
 */
internal class LightControllerImpl constructor(
    context: Context,
    private val replyHandler: ReplyHandler
) : LightController {

    private var bluetoothController: BluetoothController? = null
    private var deviceInfo: DeviceInfo? = null
    private lateinit var commandGenerator: CommandGenerator

    init {
        bluetoothController =
            BluetoothControllerImpl(context, object : BluetoothControllerCallback {
                override fun onAddDevice(device: LightDevice) {
                    replyHandler.onAddDevice(device)
                }

                override fun onConnectionFailed() {
                    deviceInfo = null
                    replyHandler.onConnectionFailed()
                }

                override fun onConnectionEstablished() {
                    commandGenerator = LightCmdGeneratorImpl()
                    replyHandler.onConnectionEstablished()
                }

                override fun onNotificationReceived(bytes: ByteArray) {
                    replyHandler.onNotificationReceived(bytes)
                }
            }, PivoDeviceUUIDs)
    }

    override fun scan() {
        bluetoothController?.scan()
    }

    override fun connectTo(device: LightDevice) {
        bluetoothController?.connectTo(device)
    }

    override fun stopScan() {
        bluetoothController?.cancelScan()
    }

    override fun disconnect() {
        bluetoothController?.disconnect()
    }

    override fun isConnected(): Boolean {
        return bluetoothController?.isConnected() ?: false
    }

    override fun getName() = bluetoothController?.getDevice()?.getName()


    override fun changeName(name: String) {
        commandGenerator.getChangeNameBytes(name)?.let { bluetoothController?.write(it) }
    }

    override fun getDeviceInfo() {
        bluetoothController?.write(commandGenerator.getDeviceInfoBytes())
    }

    override fun requestBatteryLevel() {
        bluetoothController?.write(commandGenerator.getBatteryLevelBytes())
    }

    override fun requestMacAddress() {
        bluetoothController?.write(commandGenerator.getMacAddressBytes())
    }

    override fun requestSerialNumber() {
        bluetoothController?.write(commandGenerator.getSerialNumberBytes())
    }

    override fun stopLightPreset() {
        bluetoothController?.write(commandGenerator.getStopLightPresetBytes())
    }

    override fun runLightPreset(
        brightness: Brightness,
        interval: Double,
        patternType: LightPattern
    ) {
        commandGenerator.getRunLightPresetBytes(
            brightness,
            interval,
            patternType
        )?.let {
            bluetoothController?.write(
                it
            )
        }
    }

    override fun getBrightnessLevel() {
        bluetoothController?.write(commandGenerator.getBrightnessLevelBytes())
    }

    override fun turnOffLight() {
        bluetoothController?.write(commandGenerator.getTurnOffLightBytes())
    }

    override fun turnOnLight(brightness: Brightness) {
        commandGenerator.getTurnOnLightBytes(brightness)?.let { bluetoothController?.write(it) }
    }

    override fun dimSmoothly(brightness: Brightness) {
        commandGenerator.getDimSmoothlyBytes(brightness)?.let { bluetoothController?.write(it) }
    }

    override fun flash1X(brightness: Brightness) {
        commandGenerator.getFlash1XBytes(brightness)?.let { bluetoothController?.write(it) }
    }
}